package sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler;

import com.neovisionaries.i18n.LanguageCode;
import java.util.List;
import java.util.Optional;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectQuery;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingAttribute;
import sk.kosice.konto.kkmessageservice.repository.rsql.collation.MonolingualCollationContext;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.domain.ImmutableJoin;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.domain.Join;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqColumnInfo;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqJoinableColumnInfo;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadata;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataConfigProvider;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataImpl;

public class JooqOrderingSelectQueryHandler<T extends ListingAttribute>
    implements JooqOneParamSelectQueryHandler<SelectQuery> {

  private static final Logger log = LoggerFactory.getLogger(JooqOrderingSelectQueryHandler.class);

  private final List<ListOrdering<T>> orderings;
  private final JooqListingMetadataConfigHolder metadataConfigHolder;
  private final LanguageCode[] languageCodes;

  public JooqOrderingSelectQueryHandler(
      List<ListOrdering<T>> orderings,
      JooqRsqlMetadataConfigProvider rsqlParserConfigProvider,
      LanguageCode... languageCodes) {
    this.orderings = orderings;
    this.metadataConfigHolder = new JooqListingMetadataConfigHolder(rsqlParserConfigProvider);
    this.languageCodes = languageCodes;
  }

  @Override
  public SelectQuery handle(SelectQuery query) {
    if (!orderings.isEmpty()) {
      orderings.forEach(
          ordering -> {
            if (ordering.attribute().isTranslated()) {
              addTranslatedOrderBy(ordering, query);
            } else {
              addOrderBy(ordering, query);
            }
            addJoin(ordering, query);
          });
    } else {
      log.debug(
          "Adjusting select with Ordering skipped. There is not any ordering in query defined.");
    }
    return query;
  }

  private void addOrderBy(ListOrdering<T> ordering, SelectQuery query) {
    final JooqColumnInfo<Object> columnInfo = (JooqColumnInfo<Object>) getJooqColumnInfo(ordering);

    if (columnInfo.getField().getType().equals(String.class)) {
      final JooqColumnInfo<String> stringColumnInfo =
          (JooqColumnInfo<String>) getJooqColumnInfo(ordering);
      query.addOrderBy(
          ordering.ascending()
              ? DSL.upper(stringColumnInfo.getField()).asc().nullsLast()
              : DSL.upper(stringColumnInfo.getField()).desc().nullsFirst());
    } else {
      query.addOrderBy(
          ordering.ascending()
              ? columnInfo.getField().asc().nullsLast()
              : columnInfo.getField().desc().nullsFirst());
    }
  }

  private void addTranslatedOrderBy(ListOrdering<T> ordering, SelectQuery query) {
    final JooqColumnInfo<Object> columnInfo = (JooqColumnInfo<Object>) getJooqColumnInfo(ordering);

    if (columnInfo.getField().getType().equals(String.class)) {
      final JooqColumnInfo<String> stringColumnInfo =
          (JooqColumnInfo<String>) getJooqColumnInfo(ordering);
      final Field<?> collation =
          MonolingualCollationContext.getCollationTemplate(
              query.configuration().dialect(),
              DSL.upper(stringColumnInfo.getField()),
              ordering.attribute(),
              languageCodes);

      query.addOrderBy(
          ordering.ascending() ? collation.asc().nullsLast() : collation.desc().nullsFirst());
    } else {
      final Field<?> collation =
          MonolingualCollationContext.getCollationTemplate(
              query.configuration().dialect(),
              columnInfo.getField(),
              ordering.attribute(),
              languageCodes);

      query.addOrderBy(
          ordering.ascending() ? collation.asc().nullsLast() : collation.desc().nullsFirst());
    }
  }

  private void addJoin(ListOrdering<T> ordering, SelectQuery query) {
    if (getJooqColumnInfo(ordering) instanceof JooqJoinableColumnInfo) {

      final JooqJoinableColumnInfo<Record, Object> columnInfo =
          (JooqJoinableColumnInfo<Record, Object>) getJooqColumnInfo(ordering);

      final Join join =
          ImmutableJoin.builder()
              .table(columnInfo.getTable())
              .type(columnInfo.getJoinType())
              .condition(columnInfo.getCondition())
              .build();

      query.addJoin(join.table(), join.type(), join.condition());
    }
  }

  private JooqColumnInfo<?> getJooqColumnInfo(ListOrdering<T> ordering) {
    final Optional<JooqColumnInfo<?>> maybe =
        metadataConfigHolder.metadata.findByName(ordering.attribute().apiName());

    if (maybe.isEmpty()) {
      throw new IllegalArgumentException("Unknown property: " + ordering.attribute().apiName());
    }
    return maybe.get();
  }

  static class JooqListingMetadataConfigHolder {

    private JooqRsqlMetadata metadata;

    private JooqListingMetadataConfigHolder(JooqRsqlMetadataConfigProvider provider) {
      this.metadata = new JooqRsqlMetadataImpl();
      provider.configure(this.metadata);
    }
  }
}
