package sk.kosice.konto.kkmessageservice.organization;

import static sk.kosice.konto.kkmessageservice.repository.model.Tables.RECIPIENT;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.SUBSCRIPTION;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.TOPIC;

import io.restassured.http.ContentType;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.ServiceAppFeatureSpec;
import sk.kosice.konto.kkmessageservice.restapi.controller.OrganizationController;
import sk.kosice.konto.kkmessageservice.restapi.controller.SubscriptionController;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionCreateByOrganizationIdRequest;

public class OrganizationFeatureSpec extends ServiceAppFeatureSpec
    implements OrganizationFeatureSpecTestSupport {
  protected <T> T listOrganizations(HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .get(OrganizationController.ORGANIZATIONS_URI)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T listOrganizationsWithoutJwt(HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .get(OrganizationController.ORGANIZATIONS_URI)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T listOrganizationsByKid(HttpStatus status, Class<T> response, UUID kid) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .get(OrganizationController.ORGANIZATIONS_BY_KID_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T createSubscriptions(
      HttpStatus status,
      UUID kid,
      Class<T> response,
      SubscriptionCreateByOrganizationIdRequest request) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .body(request)
        .post(SubscriptionController.SUBSCRIPTIONS_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected List<UUID> prepareTopics(UUID organizationId) {
    final var topic1 = UUID.randomUUID();
    final var topic2 = UUID.randomUUID();
    final var topic3 = UUID.randomUUID();

    var increment = 1;
    for (final var topic : List.of(topic1, topic2, topic3)) {
      dslContext
          .insertInto(TOPIC)
          .values(
              topic,
              "topic" + increment++,
              "topic" + increment++,
              organizationId,
              "organizationName")
          .execute();
    }

    return List.of(topic1, topic2, topic3);
  }

  protected UUID prepareRecipient() {
    final var kid = UUID.randomUUID();

    dslContext.insertInto(RECIPIENT).values(kid, "test").execute();

    return kid;
  }

  protected void prepareRecipient(UUID id) {
    dslContext.insertInto(RECIPIENT).values(id, "test").execute();
  }

  protected List<UUID> prepareSubscriptions(UUID recipientId, UUID... topicIds) {
    List<UUID> ids = new ArrayList<>();

    for (final var topic : List.of(topicIds)) {
      var id = UUID.randomUUID();
      dslContext.insertInto(SUBSCRIPTION).values(id, topic, recipientId, false).execute();

      ids.add(id);
    }

    return ids;
  }
}
