package sk.kosice.konto.kkmessageservice.topic;

import static org.assertj.core.api.Assertions.assertThat;
import static sk.kosice.konto.kkmessageservice.domain.common.error.ErrorCodeType.UNAUTHORIZED;
import static sk.kosice.konto.kkmessageservice.domain.permission.error.PermissionErrorCode.ACTION_IS_FORBIDDEN;
import static sk.kosice.konto.kkmessageservice.domain.topic.error.TopicErrorCode.TOPIC_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.topic.TopicApiFixtures.createTopicRequest;

import java.util.UUID;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;

public class TopicDeleteFeatureSpec extends TopicFeatureSpec {

  @Test
  public void thatTopicCanBeDeletedWithEmployeeToken() {
    final var request = createTopicRequest();
    final var response = createTopic(ORGANIZATION_ID_1, request);

    assertThat(response.getName()).isEqualTo(request.getName());
    assertThat(response.getDescription()).isEqualTo(request.getDescription());

    deleteTopic(response.getOrganizationId(), response.getId(), HttpStatus.NO_CONTENT);

    final var errorDetailResponse =
        getTopic(
            response.getOrganizationId(),
            response.getId(),
            HttpStatus.NOT_FOUND,
            ErrorDetailResponse.class);

    assertThat(errorDetailResponse.primaryFault().get().faultCode())
        .isEqualTo(TOPIC_DOES_NOT_EXIST.toString());
  }

  @Test
  public void thatNotExistingTopicCannotBeDeleted() {
    final var errorDetailResponse =
        deleteTopicWithError(
            ORGANIZATION_ID_1, UUID.randomUUID(), HttpStatus.NOT_FOUND, ErrorDetailResponse.class);

    assertThat(errorDetailResponse.primaryFault().get().faultCode())
        .isEqualTo(TOPIC_DOES_NOT_EXIST.toString());
  }

  @Test
  public void thatTopicCannotBeDeletedWithoutToken() {
    final var errorDetailResponse =
        deleteTopicWithoutJwt(
            UUID.randomUUID(),
            UUID.randomUUID(),
            HttpStatus.UNAUTHORIZED,
            ErrorDetailResponse.class);

    assertThat(errorDetailResponse.primaryFault().get().faultCode())
        .isEqualTo(UNAUTHORIZED.toString());
  }

  @Test
  public void thatTopicCannotBeDeletedWithEmployeeTokenWithoutPermissions() {
    prepareInvalidPermissions();
    final var errorDetailResponse =
        deleteTopicWithError(
            UUID.randomUUID(), UUID.randomUUID(), HttpStatus.FORBIDDEN, ErrorDetailResponse.class);

    assertThat(errorDetailResponse.primaryFault().get().faultCode())
        .isEqualTo(ACTION_IS_FORBIDDEN.toString());
  }
}
