package sk.kosice.konto.kkmessageservice.business.message.service;

import static sk.kosice.konto.kkmessageservice.domain.common.enumeration.Permission.ACTION_MANAGE_MESSAGE;

import jakarta.inject.Inject;
import jakarta.inject.Named;
import java.util.UUID;
import sk.kosice.konto.kkmessageservice.business.AbstractService;
import sk.kosice.konto.kkmessageservice.business.message.port.inbound.CreateMessageUseCase;
import sk.kosice.konto.kkmessageservice.business.message.port.outbound.PersistMessagePort;
import sk.kosice.konto.kkmessageservice.business.message.port.outbound.QueryMessagePort;
import sk.kosice.konto.kkmessageservice.business.organization.port.outbound.QueryOrganizationPort;
import sk.kosice.konto.kkmessageservice.business.permission.port.outbound.CheckPermissionsUseCase;
import sk.kosice.konto.kkmessageservice.domain.common.TimeProvider;
import sk.kosice.konto.kkmessageservice.domain.message.command.CreateMessageCommand;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ImmutableMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.ImmutableFindMessageByIdAndOrganizationIdQuery;
import sk.kosice.konto.kkmessageservice.domain.organization.entity.OrganizationEntity;

@Named
public class CreateMessageService extends AbstractService implements CreateMessageUseCase {

  private final PersistMessagePort persistMessagePort;
  private final QueryMessagePort queryMessagePort;
  private final QueryOrganizationPort queryOrganizationPort;
  private final CheckPermissionsUseCase checkPermissionsUseCase;

  @Inject
  public CreateMessageService(
      PersistMessagePort persistMessagePort,
      QueryMessagePort queryMessagePort,
      QueryOrganizationPort queryOrganizationPort,
      CheckPermissionsUseCase checkPermissionsUseCase) {
    this.persistMessagePort = persistMessagePort;
    this.queryMessagePort = queryMessagePort;
    this.queryOrganizationPort = queryOrganizationPort;
    this.checkPermissionsUseCase = checkPermissionsUseCase;
  }

  @Override
  public MessageEntity execute(CreateMessageCommand command) {
    checkPermission(
        checkPermissionsUseCase,
        command.actorId(),
        ACTION_MANAGE_MESSAGE,
        command.organizationId());

    final OrganizationEntity organization = queryOrganizationPort.findOne(command.organizationId());

    final MessageEntity data =
        ImmutableMessageEntity.builder()
            .id(UUID.randomUUID())
            .senderName(organization.name())
            .title(command.title())
            .topicId(command.topicId())
            .topicName("none")
            .createdAt(TimeProvider.offsetNow())
            .organizationId(command.organizationId())
            .body(command.body())
            .bodyType(command.bodyType())
            .bodyShort(command.bodyShort())
            .actions(command.actions())
            .tags(command.tags())
            .recipientKid(command.recipientKid())
            .addAllLocations(command.locations())
            .build();

    persistMessagePort.insert(data);
    return queryMessagePort.findOne(
        ImmutableFindMessageByIdAndOrganizationIdQuery.builder()
            .id(data.id())
            .organizationId(command.organizationId())
            .build());
  }
}
