package sk.kosice.konto.kkmessageservice.repository.message;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ImmutableListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.MessageListingAttribute;
import sk.kosice.konto.kkmessageservice.domain.common.listing.PageSize;
import sk.kosice.konto.kkmessageservice.domain.message.entity.BaseMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ListOfMessages;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.ImmutableMessageListingQuery;
import sk.kosice.konto.kkmessageservice.domain.recipient.entity.RecipientEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;

public class JooqMessageRepositoryListTest extends JooqMessageRepositoryTest {

  @Test
  public void thatEmptyMessageListCanBeRetrieved() {
    final ListOfMessages messages = listMessages(messageRepository);
    assertThat(messages.items()).isEmpty();
    assertThat(messages.totalCount()).isZero();
  }

  @Disabled
  @Test
  public void thatMessageListCanBeRetrieved() {
    createMessagesForFilteringAndSorting(UUID.randomUUID(), UUID.randomUUID());
    final ListOfMessages messages = listMessages(messageRepository);

    assertThat(messages.items()).isNotEmpty();
    assertThat(messages.totalCount()).isEqualTo(2);
  }

  @Test
  public void thatMessageListCanBeRetrievedByRecipientKid() {
    final UUID recipientKid = UUID.randomUUID();
    createMessagesForFilteringAndSorting(recipientKid, UUID.randomUUID());
    final ListOfMessages messages = listMessagesByRecipientKid(messageRepository, recipientKid);

    assertThat(messages.items()).isNotEmpty();
    assertThat(messages.totalCount()).isEqualTo(3);
  }

  @Test
  public void thatMessageListCanBeRetrievedByOrganizationId() {
    final UUID recipientKid = UUID.randomUUID();
    final UUID organizationId = UUID.randomUUID();
    createMessagesForFilteringAndSorting(recipientKid, organizationId);

    final ListOfMessages messages = listMessagesByOrganizationId(messageRepository, organizationId);
    assertThat(messages.items()).isNotEmpty();
    assertThat(messages.totalCount()).isEqualTo(2);
  }

  @Test
  public void thatMessageListCanBeFilteredByTitle() {
    final List<BaseMessageEntity> messages =
        createMessagesForFilteringAndSorting(UUID.randomUUID(), UUID.randomUUID());
    findWithFilter("title==\"" + messages.get(0).title() + "\"", 1);
  }

  @Test
  public void thatMessageListCanBeSortedByCreatedAt() {
    final UUID recipientKid = UUID.randomUUID();
    final List<BaseMessageEntity> messages =
        createMessagesForFilteringAndSorting(recipientKid, UUID.randomUUID());

    findSorted(
        recipientKid,
        new BaseMessageEntity[] {messages.get(0), messages.get(1), messages.get(2)},
        ImmutableListOrdering.of(MessageListingAttribute.CREATED_AT, true));

    findSorted(
        recipientKid,
        new BaseMessageEntity[] {messages.get(2), messages.get(1), messages.get(0)},
        ImmutableListOrdering.of(MessageListingAttribute.CREATED_AT, false));
  }

  @Test
  public void thatMessageListCanBePaginated() {
    final UUID recipientKid = UUID.randomUUID();
    final List<BaseMessageEntity> messages =
        createMessagesForFilteringAndSorting(recipientKid, UUID.randomUUID());

    findPage(
        recipientKid,
        1,
        1,
        ImmutableListOrdering.of(MessageListingAttribute.CREATED_AT, true),
        messages.get(0));

    findPage(
        recipientKid,
        2,
        2,
        ImmutableListOrdering.of(MessageListingAttribute.CREATED_AT, true),
        messages.get(2));

    findPage(
        recipientKid,
        3,
        1,
        ImmutableListOrdering.of(MessageListingAttribute.CREATED_AT, false),
        messages.get(2),
        messages.get(1),
        messages.get(0));
  }

  private List<BaseMessageEntity> createMessagesForFilteringAndSorting(
      UUID recipientKid, UUID organizationId) {
    final RecipientEntity recipient =
        persistAndFindRecipient(recipientRepository, prepareRecipientData(recipientKid));
    final TopicEntity topic = persistAndFindTopic(topicRepository);
    persistAndFindSubscription(subscriptionRepository, topic.id(), recipientKid);

    final MessageEntity m1 =
        persistAndFindMessage(
            messageRepository,
            prepareMessageData(topic.id()).withTitle("title1").withOrganizationId(organizationId));

    final MessageEntity m2 =
        persistAndFindMessage(
            messageRepository,
            prepareMessageData(topic.id(), recipient.kid())
                .withTitle("title2")
                .withOrganizationId(UUID.randomUUID()));

    final MessageEntity m3 =
        persistAndFindMessage(
            messageRepository,
            prepareMessageData(topic.id()).withTitle("title3").withOrganizationId(organizationId));

    return List.of(BaseMessageEntity.of(m1), BaseMessageEntity.of(m2), BaseMessageEntity.of(m3));
  }

  private void findWithFilter(String rsqlQuery, int expectedNumOfResults) {
    final ListOfMessages listOfMessages =
        messageRepository.list(
            ImmutableMessageListingQuery.builder()
                .pageSize(PageSize.of(100, 100))
                .addAllOrderings(Collections.emptyList())
                .rsqlQuery(Optional.of(rsqlParser.parse(rsqlQuery)))
                .build());

    assertThat(listOfMessages.totalCount()).isEqualTo(expectedNumOfResults);
  }

  @SafeVarargs
  private void findSorted(
      UUID recipientKid,
      BaseMessageEntity[] expectedOrder,
      ListOrdering<MessageListingAttribute>... orderings) {
    final ListOfMessages listOfMessages =
        messageRepository.list(
            ImmutableMessageListingQuery.builder()
                .pageSize(PageSize.of(100, 100))
                .addAllOrderings(List.of(orderings))
                .rsqlQuery(Optional.empty())
                .recipientKid(Optional.ofNullable(recipientKid))
                .build());

    assertThat(listOfMessages.items()).containsExactly(expectedOrder);
  }

  private void findPage(
      UUID recipientKid,
      int pageSize,
      int pageNumber,
      ListOrdering<MessageListingAttribute> ordering,
      BaseMessageEntity... expectedOrder) {
    final ListOfMessages listOfMessages =
        messageRepository.list(
            ImmutableMessageListingQuery.builder()
                .pageSize(PageSize.of(100, pageSize))
                .rsqlQuery(Optional.empty())
                .page(pageNumber)
                .addAllOrderings(List.of(ordering))
                .recipientKid(Optional.ofNullable(recipientKid))
                .build());

    assertThat(listOfMessages.items()).containsExactly(expectedOrder);
  }
}
