package sk.kosice.konto.kkmessageservice.message;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;
import static sk.kosice.konto.kkmessageservice.restapi.common.ObjectMapperHelper.parseStringFromJsonNodeList;

import io.restassured.http.ContentType;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.IntStream;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.ServiceAppFeatureSpec;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.restapi.controller.MessageController;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.BaseMessageDto;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageCreateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageListResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageWithKidDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageWithKidListResponse;

public abstract class MessageFeatureSpec extends ServiceAppFeatureSpec
    implements MessageFeatureSpecTestSupport {

  protected <T> T getMessageByOrganization(
      UUID organizationId, UUID messageId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .get(MessageController.MESSAGE_BY_ORGANIZATION_URI, organizationId, messageId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T getMessage(UUID kid, UUID messageId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .get(MessageController.MESSAGE_URI, kid, messageId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T getMessageWithoutJwt(
      UUID kid, UUID messageId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .get(MessageController.MESSAGE_URI, kid, messageId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T listMessagesByRecipientKid(UUID kid, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .get(MessageController.MESSAGES_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected MessageListResponse listMessagesByRecipientKid(UUID kid) {
    return listMessagesByRecipientKid(kid, HttpStatus.OK, MessageListResponse.class);
  }

  protected <T> T listMessagesByOrganizationId(
      UUID organizationId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .get(MessageController.MESSAGES_BY_ORGANIZATION_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected MessageWithKidListResponse listMessagesByOrganizationId(UUID organizationId) {
    return listMessagesByOrganizationId(
        organizationId, HttpStatus.OK, MessageWithKidListResponse.class);
  }

  protected <T> T listMessagesByOrganizationIdWithoutJwt(
      UUID organizationId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .get(MessageController.MESSAGES_BY_ORGANIZATION_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected ErrorDetailResponse listMessagesByOrganizationIdWithoutJwt(UUID organizationId) {
    return listMessagesByOrganizationIdWithoutJwt(
        organizationId, HttpStatus.UNAUTHORIZED, ErrorDetailResponse.class);
  }

  protected <T> T createMessage(
      UUID kid, MessageCreateRequest request, HttpStatus httpStatus, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .headers(defaultHeadersWithOnBehalfOf())
        .body(request)
        .post(MessageController.MESSAGES_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected MessageDetailResponse createMessage(UUID kid, MessageCreateRequest request) {
    return createMessage(kid, request, HttpStatus.CREATED, MessageDetailResponse.class);
  }

  protected <T> T createMessageAppIntegration(
      UUID kid, MessageCreateRequest request, HttpStatus httpStatus, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(APP_INTEGRATION_JWT_TOKEN)
        .headers(defaultHeadersWithOnBehalfOf())
        .body(request)
        .post(MessageController.MESSAGES_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected MessageDetailResponse createMessageAppIntegration(
      UUID kid, MessageCreateRequest request) {
    return createMessageAppIntegration(
        kid, request, HttpStatus.CREATED, MessageDetailResponse.class);
  }

  protected <T> T createMessageWithoutJwt(
      UUID kid, MessageCreateRequest request, HttpStatus httpStatus, Class<T> response) {
    return requestSpecification()
        .when()
        .headers(defaultHeadersWithOnBehalfOf())
        .body(request)
        .post(MessageController.MESSAGES_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected ErrorDetailResponse createMessageWithoutJwt(UUID kid, MessageCreateRequest request) {
    return createMessageWithoutJwt(
        kid, request, HttpStatus.UNAUTHORIZED, ErrorDetailResponse.class);
  }

  protected <T> T createPublicMessage(
      UUID organizationId, MessageCreateRequest request, HttpStatus httpStatus, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .headers(defaultHeadersWithOnBehalfOf())
        .body(request)
        .post(MessageController.MESSAGES_BY_ORGANIZATION_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected MessageDetailResponse createPublicMessage(
      UUID organizationId, MessageCreateRequest request) {
    return createPublicMessage(
        organizationId, request, HttpStatus.CREATED, MessageDetailResponse.class);
  }

  protected void checkEquality(MessageEntity message, MessageDetailResponse detailResponse) {
    assertThat(detailResponse.getId()).isEqualTo(message.id());
    assertThat(detailResponse.getSenderName()).isEqualTo(message.senderName());
    assertThat(detailResponse.getTitle()).isEqualTo(message.title());
    if (detailResponse.getTags().isPresent() && message.tags().isPresent()) {
      assertThat(parseStringFromJsonNodeList(detailResponse.getTags().get()))
          .isEqualTo(message.tags().get());
    }
    assertThat(detailResponse.getBody()).isEqualTo(message.body());
    assertThat(detailResponse.getCreatedAt()).isNotNull();
  }

  protected void checkEquality(MessageEntity message, MessageWithKidDetailResponse detailResponse) {
    assertThat(detailResponse.getId()).isEqualTo(message.id());
    assertThat(detailResponse.getSenderName()).isEqualTo(message.senderName());
    assertThat(detailResponse.getTitle()).isEqualTo(message.title());
    if (detailResponse.getTags().isPresent() && message.tags().isPresent()) {
      assertThat(parseStringFromJsonNodeList(detailResponse.getTags().get()))
          .isEqualTo(message.tags().get());
    }
    assertThat(detailResponse.getBody()).isEqualTo(message.body());
    assertThat(detailResponse.getCreatedAt()).isNotNull();
    assertThat(detailResponse.getRecipientKid()).isNotNull().isNotEmpty();
  }

  protected void checkFiltering(
      UUID recipientKid, String filter, Integer expectedCount, BaseMessageDto... expectedMessages) {

    final MessageListResponse response =
        requestSpecification()
            .when()
            .auth()
            .oauth2(CITIZEN_JWT_TOKEN)
            .get(MessageController.MESSAGES_URI + filter, recipientKid)
            .then()
            .statusCode(HttpStatus.OK.value())
            .extract()
            .body()
            .as(MessageListResponse.class);

    assertThat(response.totalCount()).isEqualTo(expectedCount);
    assertThat(response.items().size()).isEqualTo(expectedCount);

    final List<BaseMessageDto> expectedMessageList = List.of(expectedMessages);
    response
        .items()
        .forEach(
            detail -> {
              final Optional<BaseMessageDto> found =
                  expectedMessageList.stream()
                      .filter(r -> detail.getId().equals(r.getId()))
                      .findFirst();
              assertThat(found.isPresent()).isTrue();
            });
  }

  protected void checkSorting(
      UUID recipientKid, String sort, Integer totalCount, BaseMessageDto... expectedMessages) {

    final MessageListResponse response =
        requestSpecification()
            .when()
            .auth()
            .oauth2(CITIZEN_JWT_TOKEN)
            .get(MessageController.MESSAGES_URI + "?sort=" + sort, recipientKid)
            .then()
            .statusCode(HttpStatus.OK.value())
            .extract()
            .body()
            .as(MessageListResponse.class);

    assertThat(response.totalCount()).isEqualTo(totalCount);
    assertThat(response.items().size()).isEqualTo(expectedMessages.length);

    final List<BaseMessageDto> expectedMessageList = List.of(expectedMessages);
    IntStream.range(0, response.items().size())
        .forEach(
            i ->
                assertThat(response.items().get(i).getId())
                    .isEqualTo(expectedMessageList.get(i).getId()));
  }
}
