import axios, { AxiosHeaders, AxiosRequestConfig } from 'axios';
import { v4 as uuidv4 } from 'uuid';
import { NextRequest } from 'next/server';
import { auth } from '@/auth';
import { getRequestError } from './error';

const services: Record<string, string | undefined> = {
  kkmessage: process.env.API_KKMESSAGE_BASE_URL,
  kk: process.env.API_KK_BASE_URL
};

const getServiceName = (path: string) => {
  const match = path.match(/^\/v\d+\/([^\/]+)/);
  return match ? match[1] : null;
};

export const baseUrl = (service: string): string => {
  const serviceUrl = services[service];

  if (!serviceUrl) {
    throw new Error(`Base url for ${service} was not found.`);
  }

  return serviceUrl;
};

export const headers = async (otherHeaders?: Record<string, string>): Promise<AxiosHeaders> => {
  const session = await auth();
  const token = session?.accessToken;
  const organizationId = session?.user?.organization?.id;

  return new AxiosHeaders({
    ...otherHeaders,
    correlationId: uuidv4(),
    Accept: 'application/json',
    'Content-Type': 'application/json',
    'X-APP-VERSION': '1.0.0',
    'X-APP-ID': process.env.API_HEADER_X_APP_ID || '',
    'X-APP-PLATFORM': process.env.API_HEADER_X_APP_PLATFORM || '',
    'X-DEVICE-ID': process.env.API_HEADER_X_DEVICE_ID || '',
    ...(organizationId ? { 'X-On-Behalf-Of': organizationId } : {}),
    ...(token ? { Authorization: `Bearer ${token}` } : {})
  });
};

const url = (request: NextRequest): string => {
  const path = request.nextUrl.pathname.replace('/api', '');
  const params = request.nextUrl.searchParams.toString();

  const serviceName = getServiceName(path);

  if (!serviceName) {
    throw new Error(`Unable to determine service name for http request from: ${path}`);
  }

  const baseUrlAddress = baseUrl(serviceName);
  return `${baseUrlAddress}${path}${params ? `?${params}` : ''}`;
};

const config = async (request: NextRequest): Promise<AxiosRequestConfig> => {
  const { method } = request;
  let data;

  try {
    data = method === 'GET' ? undefined : await request.json();
  } catch (_) {
    data = undefined;
  }

  const accept = request.headers.get('accept') || 'application/json';
  const responseType = accept.includes('application/octet-stream') ? 'arraybuffer' : 'json';

  return {
    url: url(request),
    method,
    responseType,
    headers: await headers(),
    data
  };
};

export const handler = async (request: NextRequest) => {
  try {
    const cfg = await config(request);
    const response = await axios(cfg);

    if (cfg.responseType === 'arraybuffer') {
      const data = new Uint8Array(response.data);

      const disposition = response.headers['content-disposition'];

      return new Response(data, {
        status: 200,
        headers: {
          'Content-Type': 'application/octet-stream',
          'Content-Disposition': disposition
        }
      });
    }

    return Response.json(response.data);
  } catch (err: unknown) {
    const error = getRequestError(err);
    return Response.json(error, { status: error.status });
  }
};
