package sk.kosice.konto.kknotificationservice;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.MESSAGE;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.RECIPIENT;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.SUBSCRIPTION;
import static sk.kosice.konto.kknotificationservice.restapi.controller.BaseController.EMPLOYEE_ID_CLAIM_NAME;

import com.mailjet.client.MailjetClient;
import com.mailjet.client.MailjetRequest;
import com.mailjet.client.MailjetResponse;
import com.mailjet.client.errors.MailjetException;
import io.restassured.RestAssured;
import io.restassured.http.ContentType;
import io.restassured.http.Header;
import io.restassured.http.Headers;
import io.restassured.specification.RequestSpecification;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import no.nav.security.mock.oauth2.MockOAuth2Server;
import org.jooq.DSLContext;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.web.server.LocalServerPort;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.bean.override.mockito.MockitoBean;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import sk.kosice.konto.kkmessageservice.business.organization.port.outbound.QueryOrganizationPort;
import sk.kosice.konto.kkmessageservice.domain.organization.entity.ImmutableOrganizationEntity;
import sk.kosice.konto.kkmessageservice.domain.organization.entity.OrganizationEntity;
import sk.kosice.konto.kknotificationservice.business.email.port.inbound.CreateEmailFromMessageUseCase;
import sk.kosice.konto.kknotificationservice.business.email.port.outbound.EmailSenderPort;
import sk.kosice.konto.kknotificationservice.business.message.port.inbound.FindAndSendMessageWasNotSentUseCase;
import sk.kosice.konto.kknotificationservice.business.message.port.outbound.QueryMessagePort;
import sk.kosice.konto.kknotificationservice.mailjet.configuration.properties.MailjetClientPropertiesPort;

@ExtendWith(SpringExtension.class)
@SpringBootTest(
    webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT,
    classes = KkNotificationServiceApplication.class)
@ActiveProfiles("test")
public abstract class ServiceAppFeatureSpec {

  protected static String JWT_TECHNICAL_TOKEN;

  @LocalServerPort protected int actualPort;

  @MockitoBean protected MailjetClient mailjetClient;
  @MockitoBean protected MailjetClientPropertiesPort mailjetClientPropertiesPort;
  @MockitoBean protected QueryOrganizationPort queryOrganizationPort;

  @Autowired protected EmailSenderPort emailSenderPort;

  @Autowired protected DSLContext dslContext;
  @Autowired protected FindAndSendMessageWasNotSentUseCase findAndSendMessageWasNotSentUseCase;
  @Autowired protected CreateEmailFromMessageUseCase createEmailFromMessageUseCase;
  @Autowired protected QueryMessagePort queryMessagePort;

  protected static MockOAuth2Server oAuth2Server;

  protected static final UUID ROLE_ADMIN_ID =
      UUID.fromString("58aebdd7-726a-4f0a-9841-d4cb1c00b4d1");
  protected static String EMPLOYEE_SUB_CLAIM;

  @BeforeEach
  public void preparePermissions() {
    List<OrganizationEntity> listOfRoleOrganizations =
        new ArrayList<>() {
          {
            add(
                ImmutableOrganizationEntity.builder()
                    .id(ROLE_ADMIN_ID)
                    .isCityDistrict(false)
                    .name("Admin")
                    .build());
          }
        };

    doReturn(listOfRoleOrganizations).when(queryOrganizationPort).getUserGroups(EMPLOYEE_SUB_CLAIM);
    doReturn(
            listOfRoleOrganizations.parallelStream()
                .map(OrganizationEntity::id)
                .collect(Collectors.toList()))
        .when(queryOrganizationPort)
        .getUserGroupIds(EMPLOYEE_SUB_CLAIM);
  }

  @BeforeAll
  public static void mockOAuthStart() throws UnknownHostException {
    EMPLOYEE_SUB_CLAIM = UUID.randomUUID().toString();

    oAuth2Server = new MockOAuth2Server();
    oAuth2Server.start(InetAddress.getByName("localhost"), 8080);
    JWT_TECHNICAL_TOKEN =
        oAuth2Server
            .issueToken(
                "internal/app-integration",
                EMPLOYEE_SUB_CLAIM,
                null,
                Map.of(EMPLOYEE_ID_CLAIM_NAME, EMPLOYEE_SUB_CLAIM))
            .serialize();
  }

  @AfterAll
  public static void mockOAuthStop() {
    oAuth2Server.shutdown();
  }

  @AfterEach
  public void cleanUp() {
    dslContext.deleteFrom(SUBSCRIPTION).execute();
    dslContext.deleteFrom(RECIPIENT).execute();
    dslContext.deleteFrom(MESSAGE).execute();
  }

  public void setUpMailjet() throws MailjetException {
    when(mailjetClientPropertiesPort.apiKey()).thenReturn("apiKey");
    when(mailjetClientPropertiesPort.secretKey()).thenReturn("secretKey");

    final var response = mock(MailjetResponse.class);
    when(response.getStatus()).thenReturn(200);

    doReturn(response).when(mailjetClient).post(any(MailjetRequest.class));
  }

  public static Headers defaultHeaders() {
    return new Headers(
        List.of(
            new Header("X-CORRELATION-ID", UUID.randomUUID().toString()),
            new Header("X-APP-ID", UUID.randomUUID().toString()),
            new Header("X-APP-VERSION", "1.0.0"),
            new Header("X-APP-PLATFORM", "SERVER")));
  }

  protected RequestSpecification requestSpecification() {
    return RestAssured.given()
        .log()
        .all()
        .port(actualPort)
        .accept(ContentType.JSON)
        .contentType(ContentType.JSON)
        .headers(defaultHeaders());
  }
}
