import React from 'react';
import { Dialog, Loader, Toggle } from '@/lib/idsk';
import { Organization, Page, Subscription } from '@/utils';
import { QueryHandler, Error, Empty, Pagination } from '@/components';
import useSubscriptionsEdit from './use-subscriptions-edit';

interface SubscriptionEditDialogProps {
  userId: string;
  organization: Organization | null;
  onClose: () => void;
}

const SubscriptionEditDialog: React.FC<SubscriptionEditDialogProps> = ({ userId, organization, onClose }) => {
  const { query, handleToggle, setPage, PAGE_SIZE, t } = useSubscriptionsEdit(userId, organization);

  return (
    <Dialog
      className="subscription-edit-dialog"
      opened={!!organization}
      title={`${t('edit_subscription_title')} ${organization?.name}`}
      toggleOpened={onClose}
    >
      <div className="idsk-text-body flex flex-col gap-5">
        <p>{t('edit_subscription_description')}</p>
        <QueryHandler
          query={query}
          loading={
            <div className="min-h-64 flex items-center justify-center">
              <Loader />
            </div>
          }
          error={<Error err={query.error} />}
        >
          {(data: Page<Subscription>) => (
            <div className="flex flex-col gap-5">
              <b className="block">{`${t('topics')} (${data.totalCount})`}</b>
              <ul className="flex flex-col gap-4">
                {data.items.length > 0 ? (
                  data.items.map((subscription: Subscription) => (
                    <li
                      key={subscription.id}
                      className="subscription-item flex items-center gap-5 justify-between py-2"
                    >
                      <div className="flex flex-col">
                        <b>{subscription.topic.name}</b>
                        <span>{subscription.topic.description}</span>
                      </div>
                      <Toggle
                        checked={subscription.isEmailEnabled}
                        onChange={() => handleToggle(subscription)}
                        inputSize="small"
                      />
                    </li>
                  ))
                ) : (
                  <Empty title={t('no_topics')} />
                )}
              </ul>
              {data.totalCount > PAGE_SIZE && (
                <Pagination
                  className="ml-auto"
                  page={data.page}
                  pageSize={data.pageSize}
                  totalCount={data.totalCount}
                  onPageChange={(selected) => setPage(selected)}
                />
              )}
            </div>
          )}
        </QueryHandler>
      </div>
    </Dialog>
  );
};

export default SubscriptionEditDialog;
