package sk.kosice.konto.kkmessageservice.subscription;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.RECIPIENT;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.SUBSCRIPTION;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.TOPIC;

import io.restassured.http.ContentType;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.IntStream;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.ServiceAppFeatureSpec;
import sk.kosice.konto.kkmessageservice.restapi.controller.SubscriptionController;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionBatchUpdateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionCreateByOrganizationIdRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionListResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionUpdateRequest;

public class SubscriptionFeatureSpec extends ServiceAppFeatureSpec
    implements SubscriptionFeatureSpecTestSupport {

  protected List<UUID> prepareTopics() {
    final var topic1 = UUID.randomUUID();
    final var topic2 = UUID.randomUUID();
    final var topic3 = UUID.randomUUID();

    var increment = 1;
    for (final var topic : List.of(topic1, topic2, topic3)) {
      dslContext
          .insertInto(TOPIC)
          .values(
              topic,
              "topic" + increment++,
              "topic" + increment++,
              ORGANIZATION_ID_1,
              "organizationName")
          .execute();
    }

    return List.of(topic1, topic2, topic3);
  }

  protected UUID prepareRecipient() {
    final var kid = UUID.randomUUID();

    dslContext.insertInto(RECIPIENT).values(kid, "test").execute();

    return kid;
  }

  protected void prepareRecipient(UUID id) {
    dslContext.insertInto(RECIPIENT).values(id, "test").execute();
  }

  protected List<UUID> prepareSubscriptions(UUID recipientId, UUID... topicIds) {
    List<UUID> ids = new ArrayList<>();

    for (final var topic : List.of(topicIds)) {
      var id = UUID.randomUUID();
      dslContext.insertInto(SUBSCRIPTION).values(id, topic, recipientId, false).execute();

      ids.add(id);
    }

    return ids;
  }

  protected <T> T createSubscriptions(
      HttpStatus status,
      UUID kid,
      Class<T> response,
      SubscriptionCreateByOrganizationIdRequest request) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .body(request)
        .post(SubscriptionController.SUBSCRIPTIONS_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T listSubscriptions(HttpStatus status, UUID kid, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .get(SubscriptionController.SUBSCRIPTIONS_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected void updateSubscription(
      UUID kid, UUID id, SubscriptionUpdateRequest request, HttpStatus httpStatus) {
    requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .body(request)
        .put(SubscriptionController.SUBSCRIPTION_URI, kid, id)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value());
  }

  protected <T> T updateSubscription(
      UUID kid,
      UUID id,
      SubscriptionUpdateRequest request,
      HttpStatus httpStatus,
      Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .body(request)
        .put(SubscriptionController.SUBSCRIPTION_URI, kid, id)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .extract()
        .as(response);
  }

  protected <T> T updateSubscriptionsBatch(
      UUID kid,
      List<SubscriptionBatchUpdateRequest> requests,
      HttpStatus httpStatus,
      Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .body(requests)
        .put(SubscriptionController.SUBSCRIPTIONS_URI, kid)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected void deleteSubscriptionsByOrganizationIdAndKid(
      UUID kid, UUID organizationId, HttpStatus httpStatus) {
    requestSpecification()
        .when()
        .auth()
        .oauth2(CITIZEN_JWT_TOKEN)
        .delete(
            SubscriptionController.SUBSCRIPTIONS_URI + "?filter=organizationId==" + organizationId,
            kid)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value());
  }

  protected ErrorDetailResponse deleteSubscriptionsByOrganizationIdAndKidWithoutJwt(
      UUID kid, UUID organizationId, HttpStatus httpStatus) {
    return requestSpecification()
        .when()
        .delete(
            SubscriptionController.SUBSCRIPTIONS_URI + "?filter=organizationId==" + organizationId,
            kid)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .extract()
        .body()
        .as(ErrorDetailResponse.class);
  }

  protected List<SubscriptionDetailResponse> updateSubscriptionsBatch(
      UUID kid, List<SubscriptionBatchUpdateRequest> requests, HttpStatus httpStatus) {
    return List.of(
        updateSubscriptionsBatch(kid, requests, httpStatus, SubscriptionDetailResponse[].class));
  }

  protected void checkFiltering(
      UUID recipientKid,
      String filter,
      Integer expectedCount,
      SubscriptionDetailResponse... expected) {

    final SubscriptionListResponse response =
        requestSpecification()
            .when()
            .auth()
            .oauth2(CITIZEN_JWT_TOKEN)
            .get(SubscriptionController.SUBSCRIPTIONS_URI + filter, recipientKid)
            .then()
            .log()
            .all()
            .statusCode(HttpStatus.OK.value())
            .contentType(ContentType.JSON)
            .extract()
            .as(SubscriptionListResponse.class);

    assertThat(response.totalCount()).isEqualTo(expectedCount);
    assertThat(response.items().size()).isEqualTo(expectedCount);

    final List<SubscriptionDetailResponse> expectedMessageList = List.of(expected);
    response
        .items()
        .forEach(
            detail -> {
              final Optional<SubscriptionDetailResponse> found =
                  expectedMessageList.stream()
                      .filter(r -> detail.getId().equals(r.getId()))
                      .findFirst();
              assertThat(found.isPresent()).isTrue();
            });
  }

  protected void checkSorting(
      UUID recipientKid,
      String sort,
      Integer totalCount,
      SubscriptionDetailResponse... expectedMessages) {

    final SubscriptionListResponse response =
        requestSpecification()
            .when()
            .auth()
            .oauth2(CITIZEN_JWT_TOKEN)
            .get(SubscriptionController.SUBSCRIPTIONS_URI + "?sort=" + sort, recipientKid)
            .then()
            .log()
            .all()
            .statusCode(HttpStatus.OK.value())
            .contentType(ContentType.JSON)
            .extract()
            .as(SubscriptionListResponse.class);

    assertThat(response.totalCount()).isEqualTo(totalCount);
    assertThat(response.items().size()).isEqualTo(expectedMessages.length);

    final List<SubscriptionDetailResponse> expectedMessageList = List.of(expectedMessages);
    IntStream.range(0, response.items().size())
        .forEach(
            i ->
                assertThat(response.items().get(i).getId())
                    .isEqualTo(expectedMessageList.get(i).getId()));
  }
}
