'use client';

import React, { useMemo, useState } from 'react';
import axios from 'axios';
import { useLocale, useTranslations } from 'next-intl';
import { useQuery } from '@tanstack/react-query';
import { Loader, Table, TableRow, TableRowValue } from '@/lib/idsk';
import {
  QueryHandler,
  TableSection,
  Error,
  Pagination,
  TableList,
  TableListItem,
  TableListValue,
  Empty
} from '@/components';
import { api, unwrap, Family as FamilyDto, RequestError, date, address as buildAddress } from '@/utils';

const PAGE_SIZE = 10;

interface FamilyProps {
  session: any;
  view: 'table' | 'list';
}

interface FamilyEntryProps {
  profile: FamilyDto['profiles'][0];
  locale: string;
  view: 'list' | 'table';
}

const FamilyEntry: React.FC<FamilyEntryProps> = ({ profile, locale, view }) => {
  const t = useTranslations('my_data');

  const address = useMemo(() => {
    const permanentAddress = profile.addresses.find((item) => item.type === 'permanent');
    return permanentAddress ? buildAddress(permanentAddress) : null;
  }, [profile]);

  if (view === 'table') {
    return (
      <TableRow key={profile.identity.kid}>
        <TableRowValue>{`${profile.identity.givenName} ${profile.identity.surname}`}</TableRowValue>
        <TableRowValue>{date(profile.identity.dateOfBird, locale)}</TableRowValue>
        <TableRowValue align="right">{address || '-'}</TableRowValue>
      </TableRow>
    );
  }

  return (
    <TableListItem>
      <TableListValue label={t('name')}>{`${profile.identity.givenName} ${profile.identity.surname}`}</TableListValue>
      <TableListValue label={t('date_of_birth')}>{date(profile.identity.dateOfBird, locale)}</TableListValue>
      <TableListValue label={t('permanent_address')}>{address || '-'}</TableListValue>
    </TableListItem>
  );
};

const Family: React.FC<FamilyProps> = ({ session, view }) => {
  const [page, setPage] = useState(1);

  const t = useTranslations('my_data');
  const locale = useLocale();
  const { data } = session;

  const userId = data?.user.id || '';

  const query = useQuery<FamilyDto, RequestError>({
    queryKey: ['family', page],
    queryFn: () =>
      unwrap(
        axios.get(api.family(userId), {
          params: {
            start: page - 1,
            limit: page * PAGE_SIZE,
            conn: 'undefined'
          }
        })
      )
  });

  return (
    <TableSection title={t('family')} description={t('family_description')}>
      <QueryHandler
        query={query}
        loading={
          <div className="px-6 py-12 flex items-center justify-center w-full">
            <Loader />
          </div>
        }
        error={<Error err={query.error} />}
      >
        {(familyDto: FamilyDto) =>
          familyDto.profiles.length > 0 ? (
            <>
              {view === 'table' ? (
                <Table
                  headRowBordered
                  headRow={
                    <>
                      <TableRowValue>{t('name')}</TableRowValue>
                      <TableRowValue>{t('date_of_birth')}</TableRowValue>
                      <TableRowValue align="right">{t('permanent_address')}</TableRowValue>
                    </>
                  }
                >
                  {familyDto.profiles.map((profile) => (
                    <FamilyEntry key={profile.identity.kid} profile={profile} locale={locale} view={view} />
                  ))}
                </Table>
              ) : (
                <TableList>
                  {familyDto.profiles.map((profile) => (
                    <FamilyEntry key={profile.identity.kid} profile={profile} locale={locale} view={view} />
                  ))}
                </TableList>
              )}
              {familyDto.pagination.size > PAGE_SIZE && (
                <Pagination
                  className="ml-auto"
                  page={page}
                  pageSize={PAGE_SIZE}
                  totalCount={familyDto.pagination.size}
                  onPageChange={(selected) => setPage(selected)}
                />
              )}
            </>
          ) : (
            <Empty title={t('empty_family')} asSection />
          )
        }
      </QueryHandler>
    </TableSection>
  );
};

export default Family;
