'use client';

import axios from 'axios';
import React, { useEffect } from 'react';
import { DataGrid, DataGridRow, DataGridRowValue, PeopleIcon, PersonIcon } from '@/lib/idsk';
import { useQuery } from '@tanstack/react-query';
import { useSession } from 'next-auth/react';
import { date, Message, Page, unwrap, MessageTag as MessageTagType, api, RequestError } from '@/utils';
import { Empty, Error, QueryHandler } from '@/components';
import MessageListLoading from './MessageListSkeleton';
import MessageTag from './MessageTag';
import MessageListHeader from './MessageListHeader';
import MessageListPlaceholder from './MessageListPlaceholder';
import { useRouter } from 'next/navigation';

interface MessageListProps {
  page: number;
  pageSize: number;
  search: string;
  onLoad: (data: Page<Message>) => void;
}

const MessageTags: React.FC<{ tags: MessageTagType[] }> = ({ tags }) => (
  <div className="flex gap-2.5 flex-wrap w-full tb2:w-fit tb2:justify-end">
    {tags.map((tag, index) => (
      <MessageTag size="small" key={`tag-${index + 0}`} color={tag.color}>
        {tag.value}
      </MessageTag>
    ))}
  </div>
);

const MessageList: React.FC<MessageListProps> = ({ page, pageSize, search, onLoad }) => {
  const { data } = useSession();
  const organizationId = data?.user.organization?.id as string;

  const router = useRouter();

  const query = useQuery<Page<Message>, RequestError>({
    queryKey: ['messages', page, pageSize, search],
    queryFn: () =>
      unwrap(
        axios.get(`/api${api.messages(organizationId)}`, {
          params: {
            page,
            'page-size': pageSize,
            sort: 'createdAt:',
            filter: `body=like='*${search}*',title=like='*${search}*'`
          }
        })
      )
  });

  useEffect(() => {
    if (query.status === 'success') {
      onLoad(query.data);
    }
  }, [query.status, query.data, onLoad]);

  return (
    <QueryHandler
      query={query}
      error={
        <MessageListPlaceholder>
          <Error err={query.error} transparent />
        </MessageListPlaceholder>
      }
      loading={<MessageListLoading pageSize={pageSize} />}
    >
      {(messagePage) =>
        messagePage.items.length > 0 ? (
          <DataGrid className="custom-datagrid" headRow={<MessageListHeader />}>
            {messagePage.items.map((message: Message) => (
              <DataGridRow
                key={message.id}
                className="grid gap-1 tb2:gap-4 grid-cols-1 tb2:grid-cols-12"
                buttonProps={{ onClick: () => router.push(`/spravy/${message.id}`) }}
              >
                <DataGridRowValue className="col-span-2">
                  <span className="font-bold tb2:font-normal flex gap-3 items-center">
                    {message.recipientKid ? (
                      <>
                        <PersonIcon className="text-primary-medium flex-shrink-0" />
                        <span>Občan</span>
                      </>
                    ) : (
                      <>
                        <PeopleIcon className="text-primary-medium flex-shrink-0" />
                        <span>Všetci</span>
                      </>
                    )}
                  </span>
                </DataGridRowValue>

                <DataGridRowValue className="col-span-4 text-txt-grey">
                  <span className="truncate">{message.title}</span>
                </DataGridRowValue>

                <DataGridRowValue className="col-span-2">
                  <span className="truncate">{message.topic.name}</span>
                </DataGridRowValue>

                <DataGridRowValue align="right" className="col-span-2">
                  <MessageTags tags={message.tags || []} />
                </DataGridRowValue>

                <DataGridRowValue className="col-span-2 tb2:justify-end">{date(message.createdAt)}</DataGridRowValue>
              </DataGridRow>
            ))}
          </DataGrid>
        ) : (
          <MessageListPlaceholder>
            <div className="px-12 py-16 bg-white rounded-b-lg border-t border-neutral-300">
              <Empty title="Zatiaľ nemáte žiadne správy" />
            </div>
          </MessageListPlaceholder>
        )
      }
    </QueryHandler>
  );
};

export default MessageList;
