'use client';

import {
  Input,
  PrimaryButton,
  Loader,
  SendIcon,
  TextButton,
  HorizontalNavigation,
  HorizontalNavigationGroup,
  HorizontalNavigationItem,
  EditIcon,
  PreviewIcon
} from '@/lib/idsk';
import { forwardRef } from 'react';
import { Autocomplete, Error, QuillEditor, TagsInput } from '@/components';
import { cn, RequestError, Topic } from '@/utils';
import useNewMessage from './useNewMessage.hook';

type NewMessageType = 'general' | 'personal';

interface NewMessageFormProps {
  type: NewMessageType;
  onSuccess: () => void;
  variant?: 'dialog' | 'page';
}

const NewMessageForm = forwardRef<HTMLFormElement, NewMessageFormProps>(
  ({ type, onSuccess, variant = 'dialog' }, ref) => {
    const { mutation, send, user, query, filter, tags, topic, preview, message } = useNewMessage(onSuccess);

    const navigationItemClasses = (isActive: boolean) => {
      return cn({ 'font-bold !bg-primary-medium text-white': isActive });
    };

    const isLoading = mutation.isPending;

    return (
      <div className="flex flex-col gap-4">
        {mutation.status === 'error' && <Error err={mutation.error as unknown as RequestError} asBanner />}

        <form ref={ref} action={send} className="flex flex-col gap-2">
          {/* Sender */}
          <div className="idsk-text-body">
            <span>Odosielateľ:</span>
            <h4>{user?.organization?.name}</h4>
          </div>

          {/* Show KID input for personal messages */}
          {type === 'personal' && (
            <Input
              name="kid"
              type="text"
              label="KID"
              placeholder="Identifikátor košičana"
              disabled={isLoading}
              mandatory
              required
              fullWidth
            />
          )}

          {/* Subject */}
          <Input
            name="subject"
            type="text"
            label="Predmet"
            placeholder="Predmet správy"
            disabled={isLoading}
            mandatory
            required
            fullWidth
          />

          {/* Topic */}
          <Autocomplete
            label="Téma"
            placeholder="Vyhľadajte tému"
            mandatory
            required
            fullWidth
            disableFilter
            disabled={isLoading}
            options={query.data?.items || []}
            isLoading={query.isLoading}
            getKey={(item) => item.id}
            getLabel={(item) => item.name}
            onInputChange={function (value: string): void {
              filter.set(value);
            }}
            onOptionChange={function (value: Topic): void {
              topic.set(value);
            }}
          />

          {/* Tags */}
          <TagsInput
            placeholder="Pridajte štítok"
            subtitle="Po zadaní názvu štítku stlačte enter"
            name="tag"
            type="text"
            label="Štítky"
            optionalText="(nepovinné)"
            disabled={isLoading}
            onTagsChange={(items) => tags.set(items)}
          />

          {/* Show navigation for page variant */}
          {variant === 'page' && (
            <HorizontalNavigation className="mt-5">
              <HorizontalNavigationGroup dropdownOnMobile={false}>
                <HorizontalNavigationItem
                  className={navigationItemClasses(!preview.isOpen())}
                  type="button"
                  onClick={() => preview.toggleOpen(false)}
                  icon={<EditIcon />}
                >
                  Upraviť
                </HorizontalNavigationItem>
                <HorizontalNavigationItem
                  className={navigationItemClasses(preview.isOpen())}
                  type="button"
                  onClick={() => preview.toggleOpen(true)}
                  icon={<PreviewIcon />}
                >
                  Náhľad
                </HorizontalNavigationItem>
              </HorizontalNavigationGroup>
            </HorizontalNavigation>
          )}

          {/* Message Input */}
          <div className={cn({ hidden: preview.isOpen() && variant === 'page' })}>
            <QuillEditor value={message.get()} onChange={message.set} />
          </div>

          {/* Message Preview */}
          <div>
            {variant === 'dialog' && (
              <div className="bg-neutral-100 px-4 py-2.5">
                <div className="flex justify-between items-center">
                  <b>Náhľad</b>
                  <TextButton type="button" onClick={() => preview.toggleOpen()}>
                    {preview.isOpen() ? 'Skryť náhľad' : 'Zobraziť náhľad'}
                  </TextButton>
                </div>
              </div>
            )}
            {preview.isOpen() && (
              <div
                className={cn(
                  'p-4 min-h-52 message-detail',
                  { 'mt-5 bg-white border border-neutral-300 rounded-lg': variant === 'page' },
                  { 'bg-neutral-100': variant === 'dialog' }
                )}
                dangerouslySetInnerHTML={{
                  __html: message.get()
                }}
              />
            )}
          </div>

          {/* Footer */}
          <div
            className={cn('border-neutral-300 flex mt-5 gap-4 justify-end items-center', {
              'border-t px-7 pt-5 -ml-7 -mr-7': variant === 'dialog'
            })}
          >
            {isLoading && <Loader spinnerClassName="w-12" />}
            <PrimaryButton icon={<SendIcon />} className="h-fit" type="submit" disabled={isLoading}>
              Odoslať správu
            </PrimaryButton>
          </div>
        </form>
      </div>
    );
  }
);

NewMessageForm.displayName = 'NewMessageForm';

export default NewMessageForm;
