import { AxiosError, isAxiosError } from 'axios';
import { ApiError, Fault, RequestError, SimpleApiError } from './types';

const serverNotResponding = (err: AxiosError): RequestError => {
  return {
    status: 500,
    statusText: 'Internal Server Error',
    message: err.message,
    correlationId: '',
    fault: [
      {
        serviceId: 'web_app_spravca_kosice',
        faultCode: 'SERVER_NOT_RESPONDING',
        faultMessage: 'The request was made but no response was received.',
        faultMessageFormat: []
      }
    ]
  };
};

const requestSetupError = (err: AxiosError): RequestError => {
  return {
    status: 500,
    statusText: 'Internal Server Error',
    message: err.message,
    correlationId: '',
    fault: [
      {
        serviceId: 'web_app_spravca_kosice',
        faultCode: 'REQUEST_SETUP_ERROR',
        faultMessage: 'An error occurred while setting up the http request.',
        faultMessageFormat: []
      }
    ]
  };
};

const unknownError = (err: unknown): RequestError => {
  let message = 'Unknown error has occured.';

  if (typeof err === 'string') {
    message = err;
  }

  if (!!err && typeof err === 'object' && 'message' in err && typeof err.message === 'string') {
    message = err.message;
  }

  return {
    status: 500,
    statusText: 'Internal Server Error',
    message: message,
    correlationId: '',
    fault: [
      {
        serviceId: 'web_app_spravca_kosice',
        faultCode: 'UNKNOWN_ERROR',
        faultMessage: 'Unknown error has occured. Please contact technical support.',
        faultMessageFormat: []
      }
    ]
  };
};

// Check if { correlationId: xxx-xxx, fault: []} error was returned from the server
const isApiError = (data: unknown): data is ApiError => {
  if (!!data && typeof data === 'object' && 'correlationId' in data && 'fault' in data) {
    return true;
  }
  return false;
};

// Check if { correlationId: xxx-xxx, serviceId: xxx, faultCode: xxx, faultMessage: xxx, faultMessageFormat: []} error was returned from the server
const isSimpleApiError = (data: unknown): data is SimpleApiError => {
  if (
    !!data &&
    typeof data === 'object' &&
    'correlationId' in data &&
    'serviceId' in data &&
    'faultCode' in data &&
    'faultMessage' in data &&
    'faultMessageParams' in data
  ) {
    return true;
  }
  return false;
};

export const getRequestError = (err: unknown) => {
  if (isAxiosError(err)) {
    if (err.response) {
      // The request was made and the server responded with a status code that falls out of range of 2xx
      const data = err.response.data;
      const correlationId = data.correlationId || '';

      console.error(`[ERROR]: ${JSON.stringify(data)}`);

      let fault: Fault[] = [
        {
          serviceId: 'web_app_spravca_kosice',
          faultCode: 'UNKNOWN_HTTP_REQUEST_ERROR',
          faultMessage: 'Error body returned by the server can not be recognized. Please contact technical support.',
          faultMessageFormat: []
        }
      ];

      if (isApiError(data)) {
        fault = data.fault;
      } else if (isSimpleApiError(data)) {
        fault = [
          {
            serviceId: data.serviceId,
            faultCode: data.faultCode,
            faultMessage: data.faultMessage,
            faultMessageFormat: data.faultMessageParams
          }
        ];
      }

      const error: RequestError = {
        status: err.response.status,
        statusText: err.response.statusText,
        message: err.message,
        correlationId,
        fault
      };

      return error;
    } else if (err.request) {
      return serverNotResponding(err); // The request was made but no response was received
    } else {
      return requestSetupError(err); // Something happened in setting up the request that triggered an Error
    }
  } else {
    console.error(`[ERROR]: `, err);
    return unknownError(err); // Is not axios error
  }
};
