package sk.kosice.konto.kkmessageservice.restapi.controller.advice;

import static sk.kosice.konto.kkmessageservice.business.AbstractService.SERVICE_NAME;
import static sk.kosice.konto.kkmessageservice.domain.common.error.ServiceErrorCode.UNKNOWN_SERVICE_ERROR;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.BAD_JSON_FORMAT;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.BAD_REQUEST;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.INVALID_DATE_FORMAT;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.INVALID_ENUM_VALUE;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.INVALID_JWT;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.INVALID_TYPE_ID;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.INVALID_UUID_VALUE;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.JWT_TOKEN_PARSING_ERROR;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.METHOD_NOT_ALLOWED;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.MISSING_HEADER;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.MISSING_TYPE_ID;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.NOT_ACCEPTABLE;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.NOT_FOUND;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.UNSUPPORTED_CONTENT_TYPE;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.VALID_FROM_BEFORE_VALID_THRU;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;
import com.fasterxml.jackson.databind.exc.InvalidTypeIdException;
import com.fasterxml.jackson.databind.exc.UnrecognizedPropertyException;
import com.fasterxml.jackson.databind.exc.ValueInstantiationException;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.TypeMismatchException;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.security.oauth2.jwt.BadJwtException;
import org.springframework.security.oauth2.jwt.JwtValidationException;
import org.springframework.util.MimeType;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingRequestHeaderException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.server.ServerWebInputException;
import org.springframework.web.servlet.NoHandlerFoundException;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessError;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.restapi.controller.BaseController;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;

@ControllerAdvice
public class ServiceAdviceController {
  private static final Logger log = LoggerFactory.getLogger(ServiceAdviceController.class);

  /**
   * Handles {@link JsonParseException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(JsonParseException.class)
  public ResponseEntity<ErrorDetailResponse> handle(JsonParseException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(BAD_JSON_FORMAT.createError());
  }

  /**
   * Handles {@link ValueInstantiationException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(ValueInstantiationException.class)
  public ResponseEntity<ErrorDetailResponse> handle(ValueInstantiationException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(BAD_REQUEST.createError());
  }

  /**
   * Handles {@link TypeMismatchException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(TypeMismatchException.class)
  public ResponseEntity<ErrorDetailResponse> handle(TypeMismatchException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(BAD_REQUEST.createError());
  }

  /**
   * Handles {@link ServerWebInputException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(ServerWebInputException.class)
  public ResponseEntity<ErrorDetailResponse> handle(ServerWebInputException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(BAD_REQUEST.createError());
  }

  /**
   * Handles {@link BusinessException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(BusinessException.class)
  public ResponseEntity<ErrorDetailResponse> handle(BusinessException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(e.getError());
  }

  /**
   * Handles {@link Exception}.
   *
   * <p>Explicitly catches and handles general exception.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(Exception.class)
  public ResponseEntity<ErrorDetailResponse> handle(Exception e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(UNKNOWN_SERVICE_ERROR.createError());
  }

  /**
   * Handles {@link MethodArgumentTypeMismatchException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(MethodArgumentTypeMismatchException.class)
  public ResponseEntity<ErrorDetailResponse> handle(MethodArgumentTypeMismatchException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(BAD_REQUEST.createError(e.getName()));
  }

  /**
   * Handles {@link MissingRequestHeaderException}.
   *
   * <p>Explicitly catches and handles exception when required {@code RequestHeader} is missing.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(MissingRequestHeaderException.class)
  public ResponseEntity<ErrorDetailResponse> handle(MissingRequestHeaderException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(MISSING_HEADER.createError(e.getHeaderName()));
  }

  /**
   * Handles {@link HttpRequestMethodNotSupportedException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
  public ResponseEntity<ErrorDetailResponse> handle(HttpRequestMethodNotSupportedException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(
        METHOD_NOT_ALLOWED.createError(
            e.getMethod(),
            Objects.isNull(e.getSupportedMethods())
                ? "-"
                : String.join(",", e.getSupportedMethods())));
  }

  /**
   * Handles {@link MethodArgumentNotValidException}. Exception is thrown when validation on an
   * argument annotated with {@code @Valid} fails.
   *
   * <p>Explicitly catches and handles exception when validation of type parameter in {@code
   * RequestBody} fails.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(MethodArgumentNotValidException.class)
  public ResponseEntity<ErrorDetailResponse> handle(MethodArgumentNotValidException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    if (e.getMessage().contains(VALID_FROM_BEFORE_VALID_THRU.template()))
      return mapErrorToResponse(VALID_FROM_BEFORE_VALID_THRU.createError());
    return new MethodArgumentNotValidExceptionAdviceHandler(
            SERVICE_NAME,
            BAD_REQUEST.createError(e.getDetailMessageArguments()),
            UUID.randomUUID().toString())
        .handle(e);
  }

  /**
   * Handles {@link BadJwtException}. Exception is thrown when JWT token is in invalid format.
   *
   * <p>Explicitly catches and handles exception when validation of type parameter in {@code
   * RequestBody} fails.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(BadJwtException.class)
  public ResponseEntity<ErrorDetailResponse> handle(BadJwtException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(INVALID_JWT.createError(e.getCause().getMessage()));
  }

  /**
   * Handles {@link HttpMessageNotReadableException}.
   *
   * <p>Explicitly catches and handles exception when illegal enum type is entered.
   *
   * @param e exception to handle
   * @param request request from endpoint invoking the given exception
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(HttpMessageNotReadableException.class)
  public ResponseEntity<ErrorDetailResponse> handle(
      final HttpMessageNotReadableException e, final WebRequest request) {
    log.error("Handle {}", e.getClass().getName(), e);
    Throwable cause = e.getCause();
    if (cause instanceof JsonProcessingException) {
      if (cause instanceof InvalidFormatException ex) {
        // if invalid enum value is entered as key in Map, special handling is needed, because
        // IllegalEnumTypeException is replaced by InvalidFormatException
        if (ex.getTargetType().isEnum()) {
          return mapErrorToResponse(
              INVALID_ENUM_VALUE.createError(
                  ex.getValue(),
                  Arrays.stream(ex.getTargetType().getEnumConstants())
                      .map(en -> ((Enum) en).name())
                      .collect(Collectors.joining(", "))));
        } else if (ex.getTargetType().isAssignableFrom(UUID.class)) {
          return mapErrorToResponse(INVALID_UUID_VALUE.createError(ex.getValue()));
        } else if (ex.getTargetType().isAssignableFrom(LocalDate.class)) {
          return mapErrorToResponse(INVALID_DATE_FORMAT.createError(ex.getValue()));
        }
      } else if (cause instanceof UnrecognizedPropertyException ex) {
        return mapErrorToResponse(BAD_REQUEST.createError(ex.getPropertyName()));
      } else if (cause instanceof InvalidTypeIdException ex) {
        if (ex.getTypeId() != null) {
          return mapErrorToResponse(INVALID_TYPE_ID.createError(ex.getTypeId()));
        } else {
          return mapErrorToResponse(MISSING_TYPE_ID.createError());
        }
      }
    }
    log.info("Exception could not be properly handled: {}.", e.toString());
    return mapErrorToResponse(BAD_REQUEST.createError());
  }

  /**
   * Handles {@link NoHandlerFoundException}.
   *
   * <p>Explicitly catches and handles exception when no handler for a request is found. Property
   * 'spring.mvc.throw-exception-if-no-handler-found' must be set to 'true' if we want this
   * exception to be thrown.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler({NoHandlerFoundException.class})
  public ResponseEntity<ErrorDetailResponse> handle(NoHandlerFoundException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return this.mapErrorToResponse(NOT_FOUND.createError());
  }

  /**
   * Handles {@link HttpMediaTypeNotSupportedException}. Exception is thrown when a client POSTs,
   * PUTs, or PATCHes content of a type not supported by request handler.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
  public ResponseEntity<ErrorDetailResponse> handle(HttpMediaTypeNotSupportedException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(
        UNSUPPORTED_CONTENT_TYPE.createError(
            e.getContentType(),
            e.getSupportedMediaTypes().isEmpty()
                ? "-"
                : String.join(
                    ",", e.getSupportedMediaTypes().stream().map(MimeType::toString).toList())));
  }

  /**
   * Handles {@link JwtValidationException}.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(JwtValidationException.class)
  public ResponseEntity<ErrorDetailResponse> handle(JwtValidationException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(JWT_TOKEN_PARSING_ERROR.createError(e.getMessage()));
  }

  /**
   * Handles {@link HttpMediaTypeNotAcceptableException}. Exception thrown when the request handler
   * cannot generate a response that is acceptable by the client.
   *
   * @param e exception to handle
   * @return handled error as {@link ErrorDetailResponse} inside {@link ResponseEntity}
   */
  @ExceptionHandler(HttpMediaTypeNotAcceptableException.class)
  public ResponseEntity<ErrorDetailResponse> handle(HttpMediaTypeNotAcceptableException e) {
    log.error("Handle {}", e.getClass().getName(), e);
    return mapErrorToResponse(NOT_ACCEPTABLE.createError(HttpHeaders.ACCEPT));
  }

  public static ResponseEntity<ErrorDetailResponse> mapErrorToResponse(
      BusinessError businessError) {
    return BaseController.mapErrorToResponse(businessError);
  }
}
