'use client';

import React, { forwardRef, useCallback, useMemo, useRef, useState } from 'react';
import { Input, InputProps, KeyboardArrowDownIcon, useClickOutside, BaseButton } from '@/lib/idsk';

interface AutocompleteProps
  extends Omit<InputProps, 'onChange' | 'ref' | 'value' | 'icon' | 'iconPosition' | 'onClick'> {
  options: any[];
  disableFilter?: boolean;
  isLoading?: boolean;
  getLabel: (value: any) => string;
  getKey: (value: any) => string;
  onOptionChange: (value: any) => void;
  onInputChange?: (value: string) => void;
  getIcon?: (value: any) => React.ReactNode;
}

const Autocomplete = forwardRef<HTMLInputElement, AutocompleteProps>(
  (
    {
      isLoading,
      options: optionsProp,
      disableFilter,
      getKey,
      getLabel,
      getIcon,
      onOptionChange,
      onInputChange,
      ...inputProps
    },
    ref
  ) => {
    const [value, setValue] = useState<string>('');
    const [isOpen, setIsOpen] = useState<boolean>(false);

    const optionsRef = useRef<HTMLDivElement>(null);

    useClickOutside(() => setIsOpen(false), optionsRef);

    const handleChange = useCallback((e: React.ChangeEvent<HTMLInputElement>) => {
      const value = e.target.value;
      setValue(value);
      if (onInputChange) {
        onInputChange(value);
      }
      if (value && !isOpen) {
        setIsOpen(true);
      }
    }, []);

    const handleOptionChange = useCallback((option: any) => {
      setValue(getLabel(option));
      onOptionChange(option);
      setIsOpen(false);
    }, []);

    const options = useMemo(() => {
      if (value && !disableFilter) {
        return optionsProp.filter((item) => getLabel(item).toLowerCase().includes(value.toLowerCase()));
      }
      return optionsProp;
    }, [optionsProp, value, disableFilter]);

    return (
      <div className="relative">
        <Input
          ref={ref}
          value={value}
          onChange={handleChange}
          icon={<KeyboardArrowDownIcon />}
          iconPosition="right"
          onClick={() => setIsOpen((p) => !p)}
          {...inputProps}
        />
        {isOpen && (
          <div
            ref={optionsRef}
            className="absolute z-40 left-0 w-full rounded-md shadow-medium idsk-text-body max-h-64 overflow-y-auto bg-white"
          >
            {isLoading && <div className="px-5 py-3 text-neutral-600">Načítava sa...</div>}
            {!isLoading && options.length === 0 && <p className="text-neutral-600 px-5 py-3">Žiadne výsledky</p>}
            {!isLoading &&
              options.map((item) => (
                <BaseButton
                  className="px-5 py-3 hover:bg-primary-light text-left w-full flex items-center gap-4"
                  key={getKey(item)}
                  onClick={() => handleOptionChange(item)}
                  type="button"
                >
                  {!!getIcon && getIcon(item)}
                  {getLabel(item)}
                </BaseButton>
              ))}
          </div>
        )}
      </div>
    );
  }
);

Autocomplete.displayName = 'Autocomplete';

export default Autocomplete;
