package sk.kosice.konto.kkmessageservice.message;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;
import static sk.kosice.konto.kkmessageservice.domain.message.error.MessageErrorCode.MESSAGE_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.domain.organization.error.OrganizationErrorCode.ORGANIZATION_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.UNAUTHORIZED;

import java.util.UUID;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.recipient.entity.RecipientEntity;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageWithKidDetailResponse;

public class MessageDetailFeatureSpec extends MessageFeatureSpec {

  @Test
  public void thatMessageCanBeRetrieved() {
    final RecipientEntity recipient =
        persistAndFindRecipient(recipientRepository, prepareRecipientData(EXTENSION_KID_CLAIM));
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();

    final MessageEntity message =
        persistAndFindMessage(messageRepository, prepareMessageData(topicId, recipient.kid()));

    final MessageDetailResponse detailResponse =
        getMessage(recipient.kid(), message.id(), HttpStatus.OK, MessageDetailResponse.class);

    checkEquality(message, detailResponse);
  }

  @Test
  public void thatMessageCantBeRetrievedWithoutCitizenToken() {
    final ErrorDetailResponse detailResponse =
        getMessageWithoutJwt(
            UUID.randomUUID(),
            UUID.randomUUID(),
            HttpStatus.UNAUTHORIZED,
            ErrorDetailResponse.class);

    assertThat(detailResponse.primaryFault().get().faultCode())
        .isNotNull()
        .isEqualTo(UNAUTHORIZED.toString());
  }

  @Test
  public void thatNotExistingMessageCannotBeRetrieved() {
    final ErrorDetailResponse errorResponse =
        getMessage(
            UUID.fromString(EXTENSION_KID_CLAIM),
            UUID.randomUUID(),
            HttpStatus.NOT_FOUND,
            ErrorDetailResponse.class);

    assertThat(errorResponse).isNotNull();
    assertThat(errorResponse.primaryFault().get().faultCode())
        .isEqualTo(MESSAGE_DOES_NOT_EXIST.toString());
    assertThat(errorResponse.correlationId()).isNotEmpty();
  }

  @Test
  public void thatMessageCanBeRetrievedByOrganizationId() {
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();

    final MessageEntity message =
        persistAndFindMessage(
            messageRepository, prepareMessageDataWithOrganization(topicId, ORGANIZATION_ID_1));

    final MessageWithKidDetailResponse detailResponse =
        getMessageByOrganization(
            ORGANIZATION_ID_1, message.id(), HttpStatus.OK, MessageWithKidDetailResponse.class);

    checkEquality(message, detailResponse);
  }

  @Test
  public void thatMessageCannotBeRetrievedByInvalidOrganizationId() {
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();

    final MessageEntity message =
        persistAndFindMessage(
            messageRepository, prepareMessageDataWithOrganization(topicId, ORGANIZATION_ID_1));

    final ErrorDetailResponse detailResponse =
        getMessageByOrganization(
            NOT_EXISTING_ORGANIZATION_ID,
            message.id(),
            HttpStatus.NOT_FOUND,
            ErrorDetailResponse.class);

    assertThat(detailResponse).isNotNull();
    assertThat(detailResponse.primaryFault().get().faultCode())
        .isEqualTo(ORGANIZATION_DOES_NOT_EXIST.toString());
    assertThat(detailResponse.primaryFault().get().faultMessageParams().getFirst())
        .isEqualTo(NOT_EXISTING_ORGANIZATION_ID.toString());
  }
}
