package sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import org.jooq.GroupField;
import org.jooq.SelectQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListingQuery;
import sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingAttribute;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.BooleanValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.DateTimeValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.LocalDateValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.UUIDValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.builder.JooqRsqlTranslationsBuilder;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.builder.JooqRsqlWithListingMetadataBuilder;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqColumnInfo;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadata;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataAware;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataConfigProvider;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataImpl;

public class JooqTranslationQueryHandler<Q extends ListingQuery<? extends ListingAttribute>>
    implements JooqOneParamSelectQueryHandler<SelectQuery> {

  private static final Logger log = LoggerFactory.getLogger(JooqTranslationQueryHandler.class);
  private final JooqRsqlMetadataConfigProvider jooqListingMetadataConfigProvider;
  private final JooqListingMetadataConfigHolder metadataConfigHolder;
  private final List<String> translatedAttributeNames;
  private final Q query;

  public JooqTranslationQueryHandler(
      JooqRsqlMetadataConfigProvider jooqListingMetadataConfigProvider,
      Enum<? extends ListingAttribute>[] attributes,
      Q query) {
    this.jooqListingMetadataConfigProvider = jooqListingMetadataConfigProvider;
    this.metadataConfigHolder =
        new JooqListingMetadataConfigHolder(jooqListingMetadataConfigProvider);
    this.translatedAttributeNames =
        Arrays.stream(attributes)
            .filter(attr -> ((ListingAttribute) attr).isTranslated())
            .map(attr -> ((ListingAttribute) attr).apiName())
            .toList();
    this.query = query;
  }

  public SelectQuery handle(SelectQuery selectQuery) {
    final JooqRsqlWithListingMetadataBuilder<List<GroupField>> builder =
        new JooqRsqlTranslationsBuilder(translatedAttributeNames);
    configureJooqRsqlMetadataAwareBuilder(builder);

    List<GroupField> groupFields =
        new ArrayList<>(
            query.orderings().stream()
                .filter(attr -> attr.attribute().isTranslated())
                .map(attr -> (GroupField) getJooqColumnInfo(attr).getField())
                .toList());

    if (query.rsqlQuery().isPresent()) {
      groupFields.addAll(builder.build(query.rsqlQuery().get()));
    } else {
      log.debug(
          "Getting groupBy fields form filter query skipped. There is not any RSQL query defined.");
    }
    selectQuery.addGroupBy(groupFields.stream().distinct().toList());
    return selectQuery;
  }

  private JooqColumnInfo<?> getJooqColumnInfo(ListOrdering<? extends ListingAttribute> ordering) {
    final Optional<JooqColumnInfo<?>> maybe =
        metadataConfigHolder.metadata.findByName(ordering.attribute().apiName());

    if (maybe.isEmpty()) {
      throw new IllegalArgumentException("Unknown property: " + ordering.attribute().apiName());
    }
    return maybe.get();
  }

  private void configure(JooqRsqlMetadata metadata) {
    metadata
        .registerConverter(new BooleanValueConverter())
        .registerConverter(new DateTimeValueConverter())
        .registerConverter(new UUIDValueConverter())
        .registerConverter(new LocalDateValueConverter());

    jooqListingMetadataConfigProvider.configure(metadata);
  }

  private void configureJooqRsqlMetadataAwareBuilder(
      JooqRsqlMetadataAware jooqListingMetadataAware) {
    final JooqRsqlMetadata columnInfoMetadata = new JooqRsqlMetadataImpl();
    configure(columnInfoMetadata);
    jooqListingMetadataAware.setJooqRsqlMetadata(columnInfoMetadata);
  }

  static class JooqListingMetadataConfigHolder {

    private JooqRsqlMetadata metadata;

    private JooqListingMetadataConfigHolder(JooqRsqlMetadataConfigProvider provider) {
      this.metadata = new JooqRsqlMetadataImpl();
      provider.configure(this.metadata);
    }
  }
}
