package sk.kosice.konto.kkmessageservice.restapi.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.info.License;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.Arrays;
import org.springdoc.core.customizers.OpenApiCustomizer;
import org.springdoc.core.models.GroupedOpenApi;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import sk.kosice.konto.kkmessageservice.restapi.config.security.OAuthPropertiesPort;
import sk.kosice.konto.kkmessageservice.restapi.controller.MessageController;
import sk.kosice.konto.kkmessageservice.restapi.controller.OrganizationController;
import sk.kosice.konto.kkmessageservice.restapi.controller.PermissionController;
import sk.kosice.konto.kkmessageservice.restapi.controller.SubscriptionController;
import sk.kosice.konto.kkmessageservice.restapi.controller.TopicController;

@Configuration
public class ModuleWebApiConfig {
  public static final String BEARER_AUTH = "bearerAuth";

  @Bean
  ObjectMapper objectMapper(Jackson2ObjectMapperBuilder builder) {
    final ObjectMapper mapper =
        builder.featuresToEnable(SerializationFeature.WRITE_DATE_KEYS_AS_TIMESTAMPS).build();
    return mapper.registerModules(new JavaTimeModule(), new Jdk8Module()).findAndRegisterModules();
  }

  @Bean
  public OpenAPI openAPI(@Value("${application.rest-api.version:unknown}") String appVersion) {
    return new OpenAPI()
        .info(
            new Info()
                .title("KK-Message Component")
                .version(appVersion)
                .description("Open API documentation for kkmessage microservice.")
                .termsOfService("TODO")
                .license(
                    new License().name("MIT License").url("https://opensource.org/license/mit")));
  }

  @Bean
  public GroupedOpenApi groupedEmployeeV1Api(
      @Qualifier("employeeV1ApiCustomizer") OpenApiCustomizer employeeV1ApiCustomizer) {
    return GroupedOpenApi.builder()
        .group("Employee REST API")
        .pathsToMatch(
            prepareURIsForApiGroup(
                MessageController.MESSAGES_BY_ORGANIZATION_URI,
                MessageController.MESSAGE_BY_ORGANIZATION_URI,
                TopicController.TOPICS_URI,
                TopicController.TOPIC_URI,
                OrganizationController.ORGANIZATIONS_URI,
                PermissionController.PERMISSIONS_BY_USER_ID_URI))
        .addOpenApiCustomizer(employeeV1ApiCustomizer)
        .build();
  }

  @Bean
  public GroupedOpenApi groupedEmployeeAppIntegrationV1Api(
      @Qualifier("employeeAppIntegrationV1ApiCustomizer")
          OpenApiCustomizer employeeV1ApiCustomizer) {
    return GroupedOpenApi.builder()
        .group("Employee-App Integration REST API")
        .pathsToMatch(prepareURIsForApiGroup(MessageController.MESSAGES_URI))
        .addOpenApiCustomizer(employeeV1ApiCustomizer)
        .addOperationCustomizer( // not include get message since its citizen endpoint
            (operation, handlerMethod) -> {
              if (operation.getRequestBody() != null) {
                return operation;
              }
              return null;
            })
        .build();
  }

  @Bean
  public GroupedOpenApi groupedCitizenV1Api(
      @Qualifier("citizenV1ApiCustomizer") OpenApiCustomizer citizenV1ApiCustomizer) {
    return GroupedOpenApi.builder()
        .group("Citizen REST API")
        .pathsToMatch(
            prepareURIsForApiGroup(
                MessageController.MESSAGES_URI,
                MessageController.MESSAGE_URI,
                SubscriptionController.SUBSCRIPTION_URI,
                SubscriptionController.SUBSCRIPTIONS_URI,
                OrganizationController.ORGANIZATIONS_URI,
                OrganizationController.ORGANIZATIONS_BY_KID_URI))
        .addOpenApiCustomizer(citizenV1ApiCustomizer)
        .addOperationCustomizer( // not include create message since its employee/appIntegration
            // endpoint
            (operation, handlerMethod) -> {
              if (operation.getSummary().toLowerCase().contains("message")
                  && operation.getRequestBody() != null) {
                return null;
              }
              return operation;
            })
        .build();
  }

  @Bean(name = "employeeV1ApiCustomizer")
  public OpenApiCustomizer employeeV1ApiCustomizer(OAuthPropertiesPort oAuthPropertiesPort) {
    return openApi ->
        openApi
            .addSecurityItem(new SecurityRequirement().addList("openId"))
            .components(
                openApi
                    .getComponents()
                    .addSecuritySchemes(
                        BEARER_AUTH,
                        new SecurityScheme()
                            .description(
                                "OpenID by auto-configuration from /.well-known/openid-configuration")
                            .type(SecurityScheme.Type.OPENIDCONNECT)
                            .openIdConnectUrl(
                                oAuthPropertiesPort.ocEmployeeIssuerLocation() != null
                                    ? oAuthPropertiesPort.ocEmployeeIssuerLocation()
                                    : oAuthPropertiesPort
                                            .ocEmployeeJwksUrl()
                                            .replace("discovery/v2.0/keys", "v2.0")
                                        + "/.well-known/openid-configuration")
                            .in(SecurityScheme.In.HEADER)));
  }

  @Bean(name = "citizenV1ApiCustomizer")
  public OpenApiCustomizer citizenV1ApiCustomizer(OAuthPropertiesPort oAuthPropertiesPort) {
    return openApi ->
        openApi
            .addSecurityItem(new SecurityRequirement().addList("openId"))
            .components(
                openApi
                    .getComponents()
                    .addSecuritySchemes(
                        BEARER_AUTH,
                        new SecurityScheme()
                            .description(
                                "OpenID by auto-configuration from /.well-known/openid-configuration")
                            .type(SecurityScheme.Type.OPENIDCONNECT)
                            .openIdConnectUrl(
                                oAuthPropertiesPort.ocCitizenIssuerLocation() != null
                                    ? oAuthPropertiesPort.ocCitizenIssuerLocation()
                                    : oAuthPropertiesPort
                                            .ocCitizenJwksUrl()
                                            .replace("discovery/v2.0/keys", "v2.0")
                                        + "/.well-known/openid-configuration")
                            .in(SecurityScheme.In.HEADER)));
  }

  @Bean(name = "employeeAppIntegrationV1ApiCustomizer")
  public OpenApiCustomizer employeeAppIntegrationV1ApiCustomizer(
      OAuthPropertiesPort oAuthPropertiesPort) {
    return openApi ->
        openApi
            .addSecurityItem(new SecurityRequirement().addList("openId"))
            .components(
                openApi
                    .getComponents()
                    .addSecuritySchemes(
                        BEARER_AUTH,
                        new SecurityScheme()
                            .description(
                                "OpenID by auto-configuration from /.well-known/openid-configuration")
                            .type(SecurityScheme.Type.OPENIDCONNECT)
                            .openIdConnectUrl(
                                oAuthPropertiesPort.ocEmployeeIssuerLocation() != null
                                    ? oAuthPropertiesPort.ocEmployeeIssuerLocation()
                                    : oAuthPropertiesPort
                                            .ocEmployeeJwksUrl()
                                            .replace("discovery/v2.0/keys", "v2.0")
                                        + "/.well-known/openid-configuration")
                            .in(SecurityScheme.In.HEADER))
                    .addSecuritySchemes(
                        BEARER_AUTH + "Integration",
                        new SecurityScheme()
                            .description(
                                "OpenID by auto-configuration from /.well-known/openid-configuration")
                            .type(SecurityScheme.Type.OPENIDCONNECT)
                            .openIdConnectUrl(
                                oAuthPropertiesPort.ocAppIntegrationIssuerLocation() != null
                                    ? oAuthPropertiesPort.ocAppIntegrationIssuerLocation()
                                    : oAuthPropertiesPort
                                            .ocAppIntegrationJwksUrl()
                                            .replace("discovery/v2.0/keys", "v2.0")
                                        + "/.well-known/openid-configuration")
                            .in(SecurityScheme.In.HEADER)));
  }

  private static String[] prepareURIsForApiGroup(String... paths) {
    return Arrays.stream(paths)
        .map(path -> "/" + path.replaceAll("\\{\\w*\\}", "*"))
        .toArray(String[]::new);
  }
}
