package sk.kosice.konto.kkmessageservice.restapi.config.security;

import static sk.kosice.konto.kkmessageservice.restapi.controller.MessageController.MESSAGES_BY_ORGANIZATION_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.MessageController.MESSAGES_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.MessageController.MESSAGE_BY_ORGANIZATION_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.MessageController.MESSAGE_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.OrganizationController.ORGANIZATIONS_BY_KID_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.PermissionController.PERMISSIONS_BY_USER_ID_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.SubscriptionController.SUBSCRIPTIONS_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.SubscriptionController.SUBSCRIPTION_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.TopicController.TOPICS_URI;
import static sk.kosice.konto.kkmessageservice.restapi.controller.TopicController.TOPIC_URI;

import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtDecoders;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import sk.kosice.konto.kkmessageservice.domain.common.error.ErrorCode;
import sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode;
import sk.kosice.konto.kkmessageservice.restapi.controller.BaseController;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;

@Configuration
@EnableWebSecurity
public class SecurityConfiguration {

  public static final Logger log = LoggerFactory.getLogger(SecurityConfiguration.class);
  public static final String AUTHENTICATION_SCHEME_BEARER = "Bearer";

  RequestMatcher[] citizenEndpoints = {
    new AntPathRequestMatcher(MESSAGES_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(MESSAGE_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(SUBSCRIPTIONS_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(SUBSCRIPTIONS_URI, HttpMethod.PUT.name()),
    new AntPathRequestMatcher(SUBSCRIPTIONS_URI, HttpMethod.POST.name()),
    new AntPathRequestMatcher(SUBSCRIPTION_URI, HttpMethod.PUT.name()),
    new AntPathRequestMatcher(SUBSCRIPTIONS_URI, HttpMethod.DELETE.name()),
    new AntPathRequestMatcher(ORGANIZATIONS_BY_KID_URI, HttpMethod.GET.name()),
    // TODO: Temporary disabled
    // new AntPathRequestMatcher(ORGANIZATIONS_URI, HttpMethod.GET.name())

  };

  RequestMatcher[] employeeEndpoints = {
    new AntPathRequestMatcher(MESSAGES_BY_ORGANIZATION_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(MESSAGES_BY_ORGANIZATION_URI, HttpMethod.POST.name()),
    new AntPathRequestMatcher(MESSAGE_BY_ORGANIZATION_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(TOPICS_URI, HttpMethod.POST.name()),
    new AntPathRequestMatcher(TOPICS_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(TOPIC_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(TOPIC_URI, HttpMethod.PUT.name()),
    new AntPathRequestMatcher(TOPIC_URI, HttpMethod.DELETE.name()),
    new AntPathRequestMatcher(PERMISSIONS_BY_USER_ID_URI, HttpMethod.GET.name()),
  };

  RequestMatcher[] combinedEmployeeIntegrationEndpoints = {
    new AntPathRequestMatcher(MESSAGES_URI, HttpMethod.POST.name())
  };

  @Bean
  @Order(1)
  public SecurityFilterChain filterChainOAuthCitizen(
      HttpSecurity http, @Qualifier("ocCitizenJwtDecoder") JwtDecoder jwtDecoder) throws Exception {

    http.csrf(AbstractHttpConfigurer::disable)
        .cors(AbstractHttpConfigurer::disable)
        .securityMatchers((matchers) -> matchers.requestMatchers(citizenEndpoints))
        .authorizeHttpRequests(auth -> auth.requestMatchers(citizenEndpoints).authenticated())
        .oauth2ResourceServer(oauth2 -> oauth2.jwt(jwt -> jwt.decoder(jwtDecoder)));

    // Invokes when a server receives valid credentials that are not adequate to gain access.
    http.oauth2ResourceServer(
        configurer ->
            configurer.authenticationEntryPoint(
                invalidOrInsufficientCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    // Invokes when the request lacks valid authentication credentials for the target resource.
    http.exceptionHandling(
        configurer ->
            configurer.authenticationEntryPoint(
                missingCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    return http.build();
  }

  @Bean
  @Order(2)
  public SecurityFilterChain filterChainOAuthEmployee(
      HttpSecurity http, @Qualifier("ocEmployeeJwtDecoder") JwtDecoder jwtDecoder)
      throws Exception {

    http.csrf(AbstractHttpConfigurer::disable)
        .cors(AbstractHttpConfigurer::disable)
        .securityMatchers((matchers) -> matchers.requestMatchers(employeeEndpoints))
        .authorizeHttpRequests(auth -> auth.requestMatchers(employeeEndpoints).authenticated())
        .oauth2ResourceServer(oauth2 -> oauth2.jwt(jwt -> jwt.decoder(jwtDecoder)));

    // Invokes when a server receives valid credentials that are not adequate to gain access.
    http.oauth2ResourceServer(
        configurer ->
            configurer.authenticationEntryPoint(
                invalidOrInsufficientCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    // Invokes when the request lacks valid authentication credentials for the target resource.
    http.exceptionHandling(
        configurer ->
            configurer.authenticationEntryPoint(
                missingCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    return http.build();
  }

  @Bean
  @Order(3)
  public SecurityFilterChain filterChainOAuthCombinedEmployeeIntegration(
      HttpSecurity http, @Qualifier("combinedEmployeeIntegrationJwtDecoder") JwtDecoder jwtDecoder)
      throws Exception {

    http.csrf(AbstractHttpConfigurer::disable)
        .cors(AbstractHttpConfigurer::disable)
        .securityMatchers(
            (matchers) -> matchers.requestMatchers(combinedEmployeeIntegrationEndpoints))
        .authorizeHttpRequests(
            auth -> auth.requestMatchers(combinedEmployeeIntegrationEndpoints).authenticated())
        .oauth2ResourceServer(oauth2 -> oauth2.jwt(jwt -> jwt.decoder(jwtDecoder)));

    // Invokes when a server receives valid credentials that are not adequate to gain access.
    http.oauth2ResourceServer(
        configurer ->
            configurer.authenticationEntryPoint(
                invalidOrInsufficientCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    // Invokes when the request lacks valid authentication credentials for the target resource.
    http.exceptionHandling(
        configurer ->
            configurer.authenticationEntryPoint(
                missingCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    return http.build();
  }

  public AuthenticationEntryPoint invalidOrInsufficientCredentialsEntryPoint(
      String authenticationScheme) {
    return (request, response, authException) -> {
      if (authException instanceof AuthenticationException) {
        setApiErrorToHttpServletResponse(response, BaseApiErrorCode.UNAUTHORIZED)
            .setHeader(HttpHeaders.WWW_AUTHENTICATE, authenticationScheme);
      } else {
        setApiErrorToHttpServletResponse(response, BaseApiErrorCode.FORBIDDEN);
      }
    };
  }

  public AuthenticationEntryPoint missingCredentialsEntryPoint(String authenticationScheme) {
    return (request, response, authException) -> {
      setApiErrorToHttpServletResponse(response, BaseApiErrorCode.UNAUTHORIZED)
          .setHeader(HttpHeaders.WWW_AUTHENTICATE, authenticationScheme);
    };
  }

  private HttpServletResponse setApiErrorToHttpServletResponse(
      HttpServletResponse response, ErrorCode errorCode) throws IOException {
    final ResponseEntity<ErrorDetailResponse> responseEntity =
        BaseController.mapErrorToResponse(errorCode.createError());
    response.setStatus(responseEntity.getStatusCodeValue());
    response.setContentType(MediaType.APPLICATION_JSON_VALUE);
    String jsonBody = new ObjectMapper().writeValueAsString(responseEntity.getBody());
    response.getOutputStream().write(jsonBody.getBytes(StandardCharsets.UTF_8));
    return response;
  }

  @Bean(name = "ocAppIntegrationJwtDecoder")
  public JwtDecoder ocAppIntegrationJwtDecoder(OAuthPropertiesPort oAuthPropertiesPort) {
    if (oAuthPropertiesPort.ocAppIntegrationIssuerLocation() != null
        && !oAuthPropertiesPort.ocAppIntegrationIssuerLocation().isBlank()
        && !oAuthPropertiesPort.ocAppIntegrationIssuerLocation().isEmpty()) {
      return JwtDecoders.fromIssuerLocation(oAuthPropertiesPort.ocAppIntegrationIssuerLocation());
    } else {
      return NimbusJwtDecoder.withJwkSetUri(oAuthPropertiesPort.ocAppIntegrationJwksUrl()).build();
    }
  }

  @Bean(name = "ocEmployeeJwtDecoder")
  public JwtDecoder ocEmployeeJwtDecoder(OAuthPropertiesPort oAuthPropertiesPort) {
    if (oAuthPropertiesPort.ocEmployeeIssuerLocation() != null
        && !oAuthPropertiesPort.ocEmployeeIssuerLocation().isBlank()
        && !oAuthPropertiesPort.ocEmployeeIssuerLocation().isEmpty()) {
      return JwtDecoders.fromIssuerLocation(oAuthPropertiesPort.ocEmployeeIssuerLocation());
    } else {
      return NimbusJwtDecoder.withJwkSetUri(oAuthPropertiesPort.ocEmployeeJwksUrl()).build();
    }
  }

  @Bean(name = "ocCitizenJwtDecoder")
  public JwtDecoder ocCitizenJwtDecoder(OAuthPropertiesPort oAuthPropertiesPort) {
    if (oAuthPropertiesPort.ocCitizenIssuerLocation() != null
        && !oAuthPropertiesPort.ocCitizenIssuerLocation().isBlank()
        && !oAuthPropertiesPort.ocCitizenIssuerLocation().isEmpty()) {
      return JwtDecoders.fromIssuerLocation(oAuthPropertiesPort.ocCitizenIssuerLocation());
    } else {
      return NimbusJwtDecoder.withJwkSetUri(oAuthPropertiesPort.ocCitizenJwksUrl()).build();
    }
  }

  @Bean(name = "combinedEmployeeIntegrationJwtDecoder")
  public CombinedDoubleJwtDecoder combinedEmployeeIntegrationJwtDecoder(
      @Qualifier("ocEmployeeJwtDecoder") JwtDecoder ocEmployeeJwtDecoder,
      @Qualifier("ocAppIntegrationJwtDecoder") JwtDecoder ocAppIntegrationJwtDecoder) {
    return new CombinedDoubleJwtDecoder(ocEmployeeJwtDecoder, ocAppIntegrationJwtDecoder);
  }
}
