package sk.kosice.konto.kkmessageservice.restapi.controller;

import static sk.kosice.konto.kkmessageservice.business.AbstractService.SERVICE_NAME;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RestController;
import sk.kosice.konto.kkmessageservice.business.permission.port.inbound.ListPermissionUseCase;
import sk.kosice.konto.kkmessageservice.restapi.command.PermissionQueryFactory;
import sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponseItem;
import sk.kosice.konto.kkmessageservice.restapi.dto.permission.PermissionListResponse;
import sk.kosice.konto.kkmessageservice.restapi.mapper.PermissionResponseMapper;

@Tag(name = "Permission", description = "Rest API for Permissions.")
@RestController
@Validated
public class PermissionController extends BaseController {
  public static final String PERMISSIONS_BY_USER_ID_URI =
      BASE_V1_URI + "/employees/{userOid}/permissions";

  private final ListPermissionUseCase listPermissionUseCase;

  @Autowired
  public PermissionController(ListPermissionUseCase listPermissionUseCase) {
    this.listPermissionUseCase = listPermissionUseCase;
  }

  @Operation(
      summary = "List of permissions for user.",
      description = "Returns a list of permissions for user matching given conditions.")
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with permissions for user matching given conditions.",
        content = @Content(schema = @Schema(implementation = PermissionListResponse.class)))
  })
  @GetMapping(value = PERMISSIONS_BY_USER_ID_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> listByUserId(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "Unique identifier of user OID from IdP.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_USER_ID)
          UUID userId) {
    final var oidFromAuth = getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME);
    if (!oidFromAuth.equals(userId.toString())) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN)
          .body(
              ImmutableErrorDetailResponse.builder()
                  .correlationId(UUID.randomUUID().toString())
                  .addFault(
                      ImmutableErrorDetailResponseItem.builder()
                          .serviceId(SERVICE_NAME)
                          .faultCode(BaseApiErrorCode.FORBIDDEN.toString())
                          .faultMessage("You are not authorized to access this resource.")
                          .build())
                  .build());
    }

    return ResponseEntity.ok(
        PermissionResponseMapper.map(
            listPermissionUseCase.execute(PermissionQueryFactory.map(userId))));
  }
}
