package sk.kosice.konto.kknotificationservice.business.message.service;

import jakarta.inject.Inject;
import jakarta.inject.Named;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import sk.kosice.konto.kknotificationservice.business.AbstractService;
import sk.kosice.konto.kknotificationservice.business.businessconfig.BusinessConfigProvider;
import sk.kosice.konto.kknotificationservice.business.email.port.inbound.CreateEmailFromMessageUseCase;
import sk.kosice.konto.kknotificationservice.business.email.port.inbound.SendOneEmailUseCase;
import sk.kosice.konto.kknotificationservice.business.message.port.inbound.FindAndSendMessageWasNotSentUseCase;
import sk.kosice.konto.kknotificationservice.business.message.port.inbound.MarkMessageAsSentUseCase;
import sk.kosice.konto.kknotificationservice.business.message.port.outbound.QueryMessagePort;
import sk.kosice.konto.kknotificationservice.domain.message.command.ImmutableMarkMessageAsSentCommand;
import sk.kosice.konto.kknotificationservice.domain.message.entity.MessageEntity;

@Named
public class FindAndSendMessageWasNotSentService extends AbstractService
    implements FindAndSendMessageWasNotSentUseCase {
  public static final Logger log =
      LoggerFactory.getLogger(FindAndSendMessageWasNotSentService.class);

  private final QueryMessagePort queryMessagePort;
  private final BusinessConfigProvider businessConfigProvider;
  private final CreateEmailFromMessageUseCase createEmailFromMessageUseCase;
  private final SendOneEmailUseCase sendOneEmailUseCase;
  private final MarkMessageAsSentUseCase markMessageAsSentUseCase;

  @Inject
  public FindAndSendMessageWasNotSentService(
      QueryMessagePort queryMessagePort,
      BusinessConfigProvider businessConfigProvider,
      CreateEmailFromMessageUseCase createEmailFromMessageUseCase,
      SendOneEmailUseCase sendOneEmailUseCase,
      MarkMessageAsSentUseCase markMessageAsSentUseCase) {
    this.queryMessagePort = queryMessagePort;
    this.businessConfigProvider = businessConfigProvider;
    this.createEmailFromMessageUseCase = createEmailFromMessageUseCase;
    this.sendOneEmailUseCase = sendOneEmailUseCase;
    this.markMessageAsSentUseCase = markMessageAsSentUseCase;
  }

  @Transactional
  @Override
  public void execute() {
    final var messages =
        queryMessagePort.findNotSentMessages(
            businessConfigProvider.getBusinessConfig().messagesLimit());
    log.debug("Found {} messages to send", messages.size());

    for (final var message : messages) {
      sendEmail(message);
    }
  }

  @Transactional(propagation = Propagation.REQUIRES_NEW)
  protected void sendEmail(MessageEntity message) {
    try {
      markMessageAsSentUseCase.execute(
          ImmutableMarkMessageAsSentCommand.builder().messageId(message.id()).build());

      final var command = createEmailFromMessageUseCase.execute(message);
      sendOneEmailUseCase.execute(command);
    } catch (Exception e) {
      log.warn("Error while sending emails, detail: {}", e.getMessage(), e);
      TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
    }
  }
}
