'use client';

import React from 'react';
import axios from 'axios';
import { useNotifications } from '@/hooks';
import { api, Topic, unwrap } from '@/utils';
import { ErrorDialog, Loader, PrimaryButton, SecondaryButton } from '@/lib/idsk';
import { useMutation } from '@tanstack/react-query';

interface DeleteTopicDialogProps {
  topic: Topic | null;
  organizationId: string;
  onDelete: () => void;
  onClose: () => void;
}

const DeleteTopicDialog: React.FC<DeleteTopicDialogProps> = ({ topic, organizationId, onDelete, onClose }) => {
  const notify = useNotifications();

  const mutation = useMutation({
    mutationKey: ['topics'],
    mutationFn: (topicId: string) => unwrap(axios.delete(`/api${api.topicsDetail(organizationId, topicId)}`)),
    onSuccess: () => {
      onDelete();
    },
    onError: () => {
      onClose();
      notify({
        message: 'Tému sa nepodarilo odstrániť.',
        variant: 'warning'
      });
    }
  });

  const handleDelete = (topicId: string) => {
    mutation.mutate(topicId);
  };

  return (
    topic && (
      <ErrorDialog opened={!!topic} title={`Naozaj chete odstrániť tému "${topic.name}"?`}>
        <div className="w-full flex flex-col gap-3 relative">
          <PrimaryButton variant="warning" onClick={() => handleDelete(topic.id)}>
            Odstrániť
          </PrimaryButton>
          <SecondaryButton onClick={onClose}>Zatvoriť</SecondaryButton>

          {mutation.isPending && (
            <div className="bg-white w-full h-full left-0 top-0 absolute flex items-center justify-center">
              <Loader />
            </div>
          )}
        </div>
      </ErrorDialog>
    )
  );
};

export default DeleteTopicDialog;
