package sk.kosice.konto.kkmessageservice.repository.message;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static sk.kosice.konto.kkmessageservice.domain.message.error.MessageErrorCode.MESSAGE_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.repository.message.error.constraint.MessageConstraintErrorCode.MESSAGE_WITH_ID_ALREADY_EXISTS;

import java.util.UUID;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.TimeProvider;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.recipient.entity.RecipientEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;

public class JooqMessageRepositoryFindTest extends JooqMessageRepositoryTest {

  @Test
  public void thatMessageCanBeInsertedAndSelected() {
    final TopicEntity topic = persistAndFindTopic(topicRepository);
    final MessageEntity insertData = prepareMessageData(topic.id());
    final MessageEntity result = persistAndFindMessage(messageRepository, insertData);

    assertThat(result).isNotNull();
    assertThat(result.id()).isEqualTo(insertData.id());
    assertThat(result.senderName()).isEqualTo(insertData.senderName());
    assertThat(result.title()).isEqualTo(insertData.title());
    assertThat(result.createdAt()).isBeforeOrEqualTo(TimeProvider.offsetNow());
    assertThat(result.body()).isEqualTo(insertData.body());
    assertThat(result.organizationId()).isEqualTo(insertData.organizationId());
    assertThat(result.tags()).isEqualTo(insertData.tags());
    assertThat(result.recipientKid()).isEqualTo(insertData.recipientKid());
    assertThat(result.isNotificationSend()).isEqualTo(insertData.isNotificationSend());
    assertThat(result.bodyType()).isEqualTo(insertData.bodyType());
    assertThat(result.bodyShort()).isEqualTo(insertData.bodyShort());
  }

  @Test
  public void thatMessageWithDuplicatedIdCanNotBeInserted() {
    final TopicEntity topic = persistAndFindTopic(topicRepository);
    final MessageEntity insertData = prepareMessageData(topic.id());
    persistMessage(messageRepository, insertData);

    final BusinessException exception =
        assertThrows(BusinessException.class, () -> persistMessage(messageRepository, insertData));

    assertThat(exception.getError().code().toString())
        .isEqualTo(MESSAGE_WITH_ID_ALREADY_EXISTS.name());
  }

  @Test
  public void thatMessageCanBeInsertedWithExistingRecipientKidAndSelected() {
    final RecipientEntity recipient =
        persistAndFindRecipient(recipientRepository, prepareRecipientData());
    final TopicEntity topic = persistAndFindTopic(topicRepository);

    final MessageEntity message =
        persistAndFindMessage(messageRepository, prepareMessageData(topic.id(), recipient.kid()));

    assertThat(message).isNotNull();
    assertThat(message.recipientKid()).isPresent();
    assertThat(message.recipientKid().get()).isEqualTo(recipient.kid());
  }

  @Disabled
  @Test
  public void thatMessageCannotBeInsertedWithNotExistingRecipientKid() {
    final BusinessException exception =
        assertThrows(
            BusinessException.class,
            () -> persistMessage(messageRepository, prepareMessageData(UUID.randomUUID())));

    //    assertThat(exception.getError().code().toString())
    //        .isEqualTo(RECIPIENT_DOES_NOT_EXIST.toString());
  }

  @Test
  public void thatNotExistingMessageCannotBeSelected() {
    final BusinessException exception =
        assertThrows(
            BusinessException.class,
            () -> findMessage(messageRepository, FindMessageByIdQuery.of(UUID.randomUUID())));

    assertThat(exception.getError().code().toString()).isEqualTo(MESSAGE_DOES_NOT_EXIST.toString());
  }
}
