package sk.kosice.konto.kkmessageservice.restapi.controller;

import static sk.kosice.konto.kkmessageservice.business.AbstractService.SERVICE_NAME;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.INVALID_JWT;

import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.oauth2.jwt.Jwt;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessError;
import sk.kosice.konto.kkmessageservice.domain.common.error.NestedBusinessError;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponseItem;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponseItem;

public abstract class BaseController {
  public static final String HEADER_CORRELATION_ID = "correlationId";
  public static final String HEADER_X_ON_BEHALF_OF = "X-On-Behalf-Of";
  public static final String HEADER_X_APP_ID = "X-APP-ID";
  public static final String HEADER_X_APP_VERSION = "X-APP-VERSION";
  public static final String HEADER_X_PLATFORM = "X-APP-PLATFORM";

  public static final String BASE_V1_URI = "/v1/kkmessage";

  public static final String PARAM_MESSAGE_ID = "messageId";
  public static final String PARAM_KID = "kid";
  public static final String PARAM_ORGANIZATION_ID = "organizationId";
  public static final String PARAM_TOPIC_ID = "topicId";
  public static final String PARAM_SUBSCRIPTION_ID = "subscriptionId";
  public static final String PARAM_USER_ID = "userOid";

  public static final String EXTENSION_KID_CLAIM_NAME = "extension_kid";
  public static final String EMPLOYEE_ID_CLAIM_NAME = "oid";

  protected static final Logger log = LoggerFactory.getLogger(BaseController.class);

  public static ResponseEntity<ErrorDetailResponse> mapErrorToResponse(
      BusinessError businessError) {
    List<ErrorDetailResponseItem> errorResponseItems =
        new LinkedList<>() {
          {
            add(
                ImmutableErrorDetailResponseItem.builder()
                    .serviceId(SERVICE_NAME)
                    .faultCode(businessError.code().toString())
                    .faultMessage(businessError.code().template())
                    .addAllFaultMessageParams(businessError.params())
                    .build());
            addAll(
                businessError.nestedErrors().stream()
                    .map(BaseController::mapNestedErrorToResponseItem)
                    .toList());
          }
        };

    return ResponseEntity.status(businessError.code().type().getNumCode())
        .headers(headers -> headers.add(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE))
        .body(
            ImmutableErrorDetailResponse.builder()
                .correlationId(UUID.randomUUID().toString())
                .addAllFault(errorResponseItems)
                .build());
  }

  public static ErrorDetailResponseItem mapNestedErrorToResponseItem(
      NestedBusinessError nestedError) {
    return ImmutableErrorDetailResponseItem.builder()
        .serviceId(nestedError.serviceId())
        .faultCode(nestedError.faultCode())
        .faultMessage(nestedError.faultMessage())
        .addAllFaultMessageParams(nestedError.faultMessageParams())
        .build();
  }

  protected String getStringClaim(Jwt jwt, String claimName) {
    try {
      return jwt.getClaim(claimName);
    } catch (Exception e) {
      log.error("Error while getting claim {} from JWT", claimName, e);
      throw INVALID_JWT.createError(e.getMessage()).convertToException();
    }
  }
}
