package sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.TableField;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.EmptyValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.FieldValueConverter;

public final class JooqRsqlMetadataImpl implements JooqRsqlMetadata, JooqRsqlMetadataOnStep {

  private Map<String, JooqColumnInfo<?>> attributes;
  private List<FieldValueConverter> converters;

  public JooqRsqlMetadataImpl() {
    this.attributes = new LinkedHashMap<>();
    this.converters = new ArrayList<>();
  }

  @Override
  public <T> JooqRsqlMetadataImpl addColumnInfo(String name, Field<T> field) {

    addColumnInfo(name, field, findConverter(field));
    return this;
  }

  @Override
  public <T> JooqRsqlMetadataImpl addColumnInfo(
      String name, Field<T> field, FieldValueConverter<T> converter) {
    final JooqColumnInfo<T> columnInfo = new JooqColumnInfo<>();
    columnInfo.setField(field);
    columnInfo.setConverter(converter);

    addColumnInfo(name, columnInfo);
    return this;
  }

  @Override
  public JooqRsqlMetadataImpl addColumnInfo(String name, JooqColumnInfo info) {
    this.attributes.put(name, info);
    return this;
  }

  @Override
  public <R extends Record, T> JooqRsqlMetadataImpl addJoinableColumnInfo(
      String name, TableField<R, T> field) {

    addJoinableColumnInfo(name, field, findConverter(field));
    return this;
  }

  @Override
  public <R extends Record, T> JooqRsqlMetadataImpl addJoinableColumnInfo(
      String name, TableField<R, T> field, FieldValueConverter<T> converter) {
    final JooqJoinableColumnInfo<R, T> columnInfo = new JooqJoinableColumnInfo<>();
    columnInfo.setField(field);
    columnInfo.setTable(field.getTable());
    columnInfo.setConverter(converter);

    addJoinableColumnInfo(name, columnInfo);
    return this;
  }

  @Override
  public <R extends Record, T> JooqRsqlMetadataImpl addJoinableColumnInfo(
      String name, JooqJoinableColumnInfo<R, T> info) {
    this.attributes.put(name, info);
    return this;
  }

  @Override
  public JooqRsqlMetadataImpl joinOn(Condition condition, JoinType type) {
    final JooqColumnInfo columnInfo = findLastAttribute();

    if (columnInfo instanceof JooqJoinableColumnInfo) {
      ((JooqJoinableColumnInfo) columnInfo).setCondition(condition);
      ((JooqJoinableColumnInfo) columnInfo).setJoinType(type);
    } else {
      throw new IllegalStateException(
          "Join condition can be applied only for JooqJoinableColumnInfo.");
    }
    return this;
  }

  @Override
  public JooqRsqlMetadata joinOn(Condition condition) {
    joinOn(condition, JoinType.JOIN);
    return this;
  }

  @Override
  public Optional<JooqColumnInfo<?>> findByName(String selector) {
    return Optional.ofNullable(this.attributes.get(selector));
  }

  @Override
  public JooqRsqlMetadataImpl registerConverter(FieldValueConverter converter) {
    this.converters.add(converter);
    return this;
  }

  private <T> FieldValueConverter<T> findConverter(Field<T> field) {
    return this.converters.stream()
        .filter(converter -> converter.isAccessibleFor(field.getType()))
        .findFirst()
        .orElse(new EmptyValueConverter());
  }

  private JooqColumnInfo<?> findLastAttribute() {
    final Iterator<Map.Entry<String, JooqColumnInfo<?>>> iter = attributes.entrySet().iterator();
    Map.Entry<String, JooqColumnInfo<?>> entry = null;
    while (iter.hasNext()) {
      entry = iter.next();
    }
    return entry.getValue();
  }
}
