package sk.kosice.konto.kkmessageservice.repository.subscription.error.constraint;

import java.util.Collection;
import java.util.stream.Collectors;
import sk.kosice.konto.kkmessageservice.domain.common.error.ErrorCodeType;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.SubscriptionEntity;
import sk.kosice.konto.kkmessageservice.repository.error.ConstraintErrorCode;
import sk.kosice.konto.kkmessageservice.repository.model.Keys;
import sk.kosice.konto.kkmessageservice.repository.model.Tables;

public enum SubscriptionConstraintErrorCode implements ConstraintErrorCode {
  SUBSCRIPTION_WITH_ID_ALREADY_EXISTS("Subscription with id '%s' already exists.") {
    @Override
    public String constraintName() {
      return Keys.SUBSCRIPTION_PKEY.getName();
    }

    @Override
    public String[] findParams(Object dataObject) {
      if (dataObject instanceof Collection) {
        if (!((Collection<?>) dataObject).isEmpty()
            && ((Collection<?>) dataObject).stream().findFirst().orElse(null)
                instanceof SubscriptionEntity) {
          final Collection<SubscriptionEntity> listDataObject =
              (Collection<SubscriptionEntity>) dataObject;
          // duplicate id in a list
          return new String[] {
            listDataObject.stream()
                .collect(Collectors.groupingBy(SubscriptionEntity::id, Collectors.counting()))
                .entrySet()
                .stream()
                .filter(entity -> entity.getValue() > 1)
                .findFirst()
                .map(duplicities -> duplicities.getKey().toString())
                .orElse("ambiguous id")
          };
        } else {
          return new String[] {"null"};
        }
      } else {
        return new String[] {"null"};
      }
    }
  },
  SUBSCRIPTION_WITH_TOPIC_ID_AND_RECIPIENT_KID_ALREADY_EXISTS(
      "Subscription with topic id '%s' and recipient kid '%s' already exists.") {
    @Override
    public String constraintName() {
      return Keys.UNIQUE_SUBSCRIPTION_TOPIC_ID_RECIPIENT_KID.getName();
    }

    @Override
    public String[] findParams(Object dataObject) {
      if (dataObject instanceof Collection) {
        if (!((Collection<?>) dataObject).isEmpty()
            && ((Collection<?>) dataObject).stream().findFirst().orElse(null)
                instanceof SubscriptionEntity) {
          final Collection<SubscriptionEntity> listDataObject =
              (Collection<SubscriptionEntity>) dataObject;
          if (listDataObject.size() == 1) {
            final SubscriptionEntity data = listDataObject.stream().findFirst().get();
            return new String[] {data.recipientKid().toString(), data.topic().id().toString()};
          }
          return new String[] {"ambiguous recipientId", "ambiguous topicId"};
        } else {
          return new String[] {"null", "null"};
        }
      } else if (dataObject instanceof SubscriptionEntity subscription) {
        return new String[] {
          subscription.recipientKid().toString(), subscription.topic().id().toString()
        };
      } else {
        return new String[] {"null", "null"};
      }
    }
  },
  RECIPIENT_DOES_NOT_EXIST("Recipient with kid '%s' does not exist.") {
    public String constraintName() {
      return Keys.SUBSCRIPTION__SUBSCRIPTION_RECIPIENT_ID_FK.getName();
    }

    @Override
    public String[] findParams(Object dataObject) {
      if (dataObject instanceof Collection) {
        if (!((Collection<?>) dataObject).isEmpty()
            && ((Collection<?>) dataObject).stream().findFirst().orElse(null)
                instanceof SubscriptionEntity) {
          final Collection<SubscriptionEntity> listDataObject =
              (Collection<SubscriptionEntity>) dataObject;
          return (listDataObject.size() == 1)
              ? new String[] {listDataObject.stream().findFirst().get().recipientKid().toString()}
              : new String[] {"ambiguous recipientKid"};
        }
      }
      return new String[] {"null"};
    }

    @Override
    public ErrorCodeType type() {
      return ErrorCodeType.NOT_FOUND;
    }
  },
  TOPIC_DOES_NOT_EXIST("Topic with id '%s' does not exist.") {
    public String constraintName() {
      return Keys.SUBSCRIPTION__SUBSCRIPTION_TOPIC_ID_FK.getName();
    }

    @Override
    public String[] findParams(Object dataObject) {
      if (dataObject instanceof Collection) {
        if (!((Collection<?>) dataObject).isEmpty()
            && ((Collection<?>) dataObject).stream().findFirst().orElse(null)
                instanceof SubscriptionEntity) {
          final Collection<SubscriptionEntity> listDataObject =
              (Collection<SubscriptionEntity>) dataObject;
          return (listDataObject.size() == 1)
              ? new String[] {listDataObject.stream().findFirst().get().topic().id().toString()}
              : new String[] {"ambiguous topicId"};
        }
      }
      return new String[] {"null"};
    }

    @Override
    public ErrorCodeType type() {
      return ErrorCodeType.NOT_FOUND;
    }
  };

  private final String template;

  SubscriptionConstraintErrorCode(String template) {
    this.template = template;
  }

  @Override
  public String template() {
    return this.template;
  }

  @Override
  public String table() {
    return Tables.SUBSCRIPTION.getName();
  }

  @Override
  public abstract String constraintName();
}
