package sk.kosice.konto.kkmessageservice.repository.subscription;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static sk.kosice.konto.kkmessageservice.repository.error.DbErrorCode.DB_ERROR;

import java.util.List;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.BaseSubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.SubscriptionEntity;

public class JooqSubscriptionRepositoryUpdateTest extends JooqSubscriptionRepositoryTest {

  @Test
  public void thatSubscriptionsCanBeUpdatedInBatch() {
    final List<UUID> topics = prepareTopics();
    final UUID recipientKid = prepareRecipient();
    final List<UUID> subscriptions =
        prepareSubscriptions(recipientKid, topics.toArray(UUID[]::new));

    final List<BaseSubscriptionEntity> updateData =
        subscriptions.stream().map(this::prepareSubscriptionUpdateEntity).toList();
    updateSubscriptionsBatch(subscriptionRepository, updateData);
    final List<SubscriptionEntity> result = findSubscriptions(subscriptionRepository, recipientKid);

    assertThat(result).isNotNull().isNotEmpty().hasSize(3);
    result.forEach(
        subscriptionEntity -> {
          assertTrue(topics.contains(subscriptionEntity.topic().id()));
          assertThat(subscriptionEntity.recipientKid()).isEqualTo(recipientKid);
          assertThat(subscriptionEntity.isEmailEnabled()).isTrue();
        });
  }

  @Test
  public void thatSubscriptionCanBeUpdated() {
    final List<UUID> topics = prepareTopics();
    final UUID recipientKid = prepareRecipient();
    final UUID subscription =
        prepareSubscriptions(recipientKid, topics.toArray(UUID[]::new)).getFirst();

    final BaseSubscriptionEntity updateData = prepareSubscriptionUpdateEntity(subscription);
    updateSubscription(subscriptionRepository, updateData);
    final List<SubscriptionEntity> result = findSubscriptions(subscriptionRepository, recipientKid);

    assertThat(result).isNotNull().isNotEmpty().hasSize(3);
    assertThat(
            result.stream()
                .filter(r -> r.id().equals(subscription))
                .findFirst()
                .get()
                .isEmailEnabled())
        .isTrue();
  }

  @Test
  public void thatNotExistingSubscriptionsCannotBeUpdatedInBatch() {
    final var id = UUID.randomUUID();
    final BusinessException exception =
        assertThrows(
            BusinessException.class,
            () ->
                updateSubscriptionsBatch(
                    subscriptionRepository, List.of(prepareSubscriptionUpdateEntity(id))));

    assertThat(exception.getError().code()).isEqualTo(DB_ERROR);
  }

  @Test
  public void thatNotExistingSubscriptionCannotBeUpdated() {
    final var id = UUID.randomUUID();
    final BusinessException exception =
        assertThrows(
            BusinessException.class,
            () -> updateSubscription(subscriptionRepository, prepareSubscriptionUpdateEntity(id)));

    assertThat(exception.getError().code()).isEqualTo(DB_ERROR);
  }
}
