package sk.kosice.konto.kkmessageservice.restapi.controller;

import static sk.kosice.konto.kkmessageservice.business.AbstractService.SERVICE_NAME;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.*;
import static sk.kosice.konto.kkmessageservice.restapi.controller.MessageController.EXTENSION_KID_CLAIM_NAME;

import com.neovisionaries.i18n.LanguageCode;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ExampleObject;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import java.util.List;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import sk.kosice.konto.kkmessageservice.business.subscription.port.inbound.BatchDeleteSubscriptionByOrganizationIdAndKidUseCase;
import sk.kosice.konto.kkmessageservice.business.subscription.port.inbound.BatchUpdateSubscriptionUseCase;
import sk.kosice.konto.kkmessageservice.business.subscription.port.inbound.CreateSubscriptionsByOrganizationIdUseCase;
import sk.kosice.konto.kkmessageservice.business.subscription.port.inbound.ListSubscriptionUseCase;
import sk.kosice.konto.kkmessageservice.business.subscription.port.inbound.UpdateSubscriptionUseCase;
import sk.kosice.konto.kkmessageservice.domain.common.listing.SubscriptionListingAttribute;
import sk.kosice.konto.kkmessageservice.restapi.command.SubscriptionCommandFactory;
import sk.kosice.konto.kkmessageservice.restapi.command.SubscriptionQueryFactory;
import sk.kosice.konto.kkmessageservice.restapi.common.enumeration.Platform;
import sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode;
import sk.kosice.konto.kkmessageservice.restapi.common.listing.ListFilteringParser;
import sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponseItem;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionBatchUpdateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionCreateByOrganizationIdRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionUpdateRequest;
import sk.kosice.konto.kkmessageservice.restapi.mapper.SubscriptionResponseMapper;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.FilterParameter;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.SortParameter;

@Tag(name = "Subscription", description = "Subscription management")
@RestController
public class SubscriptionController extends BaseController {
  public static final String SUBSCRIPTIONS_URI = BASE_V1_URI + "/recipients/{kid}/subscriptions";
  public static final String SUBSCRIPTION_URI = SUBSCRIPTIONS_URI + "/{subscriptionId}";

  private final ListSubscriptionUseCase listSubscriptionUseCase;
  private final BatchUpdateSubscriptionUseCase batchUpdateSubscriptionUseCase;
  private final UpdateSubscriptionUseCase updateSubscriptionUseCase;
  private final BatchDeleteSubscriptionByOrganizationIdAndKidUseCase
      batchDeleteSubscriptionByOrganizationIdAndKidUseCase;
  private final CreateSubscriptionsByOrganizationIdUseCase
      createSubscriptionsByOrganizationIdUseCase;
  private final ListFilteringParser listFilteringParser;
  private final ListOrderingParser listOrderingParser;

  @Autowired
  public SubscriptionController(
      ListSubscriptionUseCase listSubscriptionUseCase,
      BatchUpdateSubscriptionUseCase batchUpdateSubscriptionUseCase,
      UpdateSubscriptionUseCase updateSubscriptionUseCase,
      BatchDeleteSubscriptionByOrganizationIdAndKidUseCase
          batchDeleteSubscriptionByOrganizationIdAndKidUseCase,
      CreateSubscriptionsByOrganizationIdUseCase createSubscriptionsByOrganizationIdUseCase,
      ListFilteringParser listFilteringParser,
      ListOrderingParser listOrderingParser) {
    this.listSubscriptionUseCase = listSubscriptionUseCase;
    this.batchUpdateSubscriptionUseCase = batchUpdateSubscriptionUseCase;
    this.updateSubscriptionUseCase = updateSubscriptionUseCase;
    this.batchDeleteSubscriptionByOrganizationIdAndKidUseCase =
        batchDeleteSubscriptionByOrganizationIdAndKidUseCase;
    this.createSubscriptionsByOrganizationIdUseCase = createSubscriptionsByOrganizationIdUseCase;
    this.listFilteringParser = listFilteringParser;
    this.listOrderingParser = listOrderingParser;
  }

  @Operation(
      summary = "List of subscriptions.",
      description = "Returns a list of subscriptions matching given conditions.")
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with subscriptions matching given conditions.",
        content =
            @Content(
                array =
                    @ArraySchema(
                        schema = @Schema(implementation = SubscriptionDetailResponse.class)))),
  })
  @GetMapping(value = SUBSCRIPTIONS_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> list(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @FilterParameter(
              listingAttributeClass = SubscriptionListingAttribute.class,
              example = "body=like=*Pozvánka*")
          @RequestParam(name = PARAM_FILTER, defaultValue = "")
          String filter,
      @Parameter(description = "Page query parameter.")
          @RequestParam(name = PARAM_PAGE, defaultValue = "1")
          Integer page,
      @Parameter(description = "Page size query parameter.")
          @RequestParam(name = PARAM_PAGE_SIZE, defaultValue = "10")
          Integer pageSize,
      @SortParameter(listingAttributeClass = SubscriptionListingAttribute.class)
          @RequestParam(name = PARAM_SORT, defaultValue = "")
          String orderings,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(value = PARAM_KID)
          UUID kid) {
    log.info("Calling SubscriptionController GET list: [{}]", SUBSCRIPTIONS_URI);

    final var kidFromAuth = getStringClaim(jwt, EXTENSION_KID_CLAIM_NAME);
    if (!kidFromAuth.equals(kid.toString())) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN)
          .body(
              ImmutableErrorDetailResponse.builder()
                  .correlationId(UUID.randomUUID().toString())
                  .addFault(
                      ImmutableErrorDetailResponseItem.builder()
                          .serviceId(SERVICE_NAME)
                          .faultCode(BaseApiErrorCode.BAD_REQUEST.toString())
                          .faultMessage("You are not authorized to access this resource.")
                          .build())
                  .build());
    }

    return ResponseEntity.ok(
        SubscriptionResponseMapper.map(
            listSubscriptionUseCase.execute(
                SubscriptionQueryFactory.map(
                    kid,
                    listFilteringParser.parse(
                        filter, List.of(SubscriptionListingAttribute.values())),
                    listOrderingParser.parse(
                        orderings,
                        List.of(SubscriptionListingAttribute.values()),
                        new LanguageCode[] {}),
                    page,
                    pageSize,
                    20))));
  }

  @Operation(
      summary = "Updates a subscription.",
      description = "Updates a subscription.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for updated subscription.",
              required = true,
              content =
                  @Content(schema = @Schema(implementation = SubscriptionUpdateRequest.class))))
  @ApiResponses({
    @ApiResponse(
        responseCode = "204",
        description = "Successful operation - response with details of the updated subscription."),
    @ApiResponse(
        responseCode = "404",
        description = "Subscription was not found",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "SUBSCRIPTION_DOES_NOT_EXIST",
                      description = "Subscription with kid does not exist.",
                      value =
                          """
{
"correlationId":"f99d9a88-589e-40aa-90fc-ddf75a9bb478",
"serviceId": "kkmessage-service",
"faultCode": "SUBSCRIPTION_DOES_NOT_EXIST",
"faultTopic": "Subscription with id '%s' does not exist.",
"faultTopicParams": [
"fd6c405f-7ae5-4d97-816e-a929c04b7001"
]
}
""")
                }))
  })
  @PutMapping(value = SUBSCRIPTION_URI)
  public ResponseEntity<?> update(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @Parameter(
              description = "Unique identifier of subscription.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_SUBSCRIPTION_ID)
          UUID subscriptionId,
      @Valid @RequestBody SubscriptionUpdateRequest request) {
    log.info(
        "Calling SubscriptionController PUT update: [{}]; request ({})", SUBSCRIPTION_URI, request);

    final var kidFromAuth = getStringClaim(jwt, EXTENSION_KID_CLAIM_NAME);
    if (!kidFromAuth.equals(kid.toString())) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN)
          .body(
              ImmutableErrorDetailResponse.builder()
                  .correlationId(UUID.randomUUID().toString())
                  .addFault(
                      ImmutableErrorDetailResponseItem.builder()
                          .serviceId(SERVICE_NAME)
                          .faultCode(BaseApiErrorCode.BAD_REQUEST.toString())
                          .faultMessage("You are not authorized to access this resource.")
                          .build())
                  .build());
    }

    updateSubscriptionUseCase.execute(SubscriptionCommandFactory.map(request, kid, subscriptionId));

    return ResponseEntity.noContent().build();
  }

  @Operation(
      summary = "Updates a subscriptions in a batch.",
      description = "Updates a subscriptions in a batch.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for updated subscriptions.",
              required = true,
              content =
                  @Content(
                      array =
                          @ArraySchema(
                              schema =
                                  @Schema(implementation = SubscriptionDetailResponse.class)))))
  @ApiResponses({
    @ApiResponse(
        responseCode = "204",
        description = "Successful operation - response with details of the updated subscription."),
    @ApiResponse(
        responseCode = "404",
        description = "Subscription was not found",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "SUBSCRIPTION_DOES_NOT_EXIST",
                      description = "Subscription with kid does not exist.",
                      value =
                          """
{
"correlationId":"f99d9a88-589e-40aa-90fc-ddf75a9bb478",
"serviceId": "kkmessage-service",
"faultCode": "SUBSCRIPTION_DOES_NOT_EXIST",
"faultTopic": "Subscription with id '%s' does not exist.",
"faultTopicParams": [
"fd6c405f-7ae5-4d97-816e-a929c04b7001"
]
}
""")
                }))
  })
  @PutMapping(value = SUBSCRIPTIONS_URI)
  public ResponseEntity<?> updateBatch(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @Valid @RequestBody List<SubscriptionBatchUpdateRequest> requests) {
    log.info(
        "Calling SubscriptionController PUT update: [{}]; request ({})",
        SUBSCRIPTIONS_URI,
        requests);

    final var kidFromAuth = getStringClaim(jwt, EXTENSION_KID_CLAIM_NAME);
    if (!kidFromAuth.equals(kid.toString())) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN)
          .body(
              ImmutableErrorDetailResponse.builder()
                  .correlationId(UUID.randomUUID().toString())
                  .addFault(
                      ImmutableErrorDetailResponseItem.builder()
                          .serviceId(SERVICE_NAME)
                          .faultCode(BaseApiErrorCode.BAD_REQUEST.toString())
                          .faultMessage("You are not authorized to access this resource.")
                          .build())
                  .build());
    }

    return new ResponseEntity<>(
        SubscriptionResponseMapper.map(
            batchUpdateSubscriptionUseCase.execute(SubscriptionCommandFactory.map(requests, kid))),
        HttpStatus.OK);
  }

  @Operation(
      summary = "Deletes a subscriptions by organization.",
      description = "Deletes a subscriptions by organization.")
  @ApiResponses({
    @ApiResponse(
        responseCode = "204",
        description = "Successful operation - subscriptions deleted."),
  })
  @DeleteMapping(value = SUBSCRIPTIONS_URI)
  public ResponseEntity<?> delete(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @FilterParameter(
              listingAttributeClass = SubscriptionListingAttribute.class,
              example = "body=like=*Pozvánka*")
          @RequestParam(name = PARAM_FILTER, defaultValue = "")
          String filter) {
    log.info("Calling SubscriptionController DELETE delete: [{}];", SUBSCRIPTIONS_URI);
    batchDeleteSubscriptionByOrganizationIdAndKidUseCase.execute(
        SubscriptionCommandFactory.map(
            kid,
            listFilteringParser.parse(filter, List.of(SubscriptionListingAttribute.values()))));

    return ResponseEntity.noContent().build();
  }

  @Operation(
      summary = "Creates a subscriptions.",
      description = "Creates a subscriptions.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for creating subscriptions.",
              required = true,
              content =
                  @Content(
                      schema =
                          @Schema(
                              implementation = SubscriptionCreateByOrganizationIdRequest.class))))
  @ApiResponses({
    @ApiResponse(
        responseCode = "201",
        description = "Successful operation - response with details of the created subscriptions.",
        content =
            @Content(
                schema =
                    @Schema(implementation = SubscriptionCreateByOrganizationIdRequest.class))),
  })
  @PostMapping(value = SUBSCRIPTIONS_URI)
  public ResponseEntity<?> create(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @Valid @RequestBody SubscriptionCreateByOrganizationIdRequest request) {
    log.info(
        "Calling SubscriptionController POST create: [{}]; request ({})",
        SUBSCRIPTIONS_URI,
        request);

    return new ResponseEntity<>(
        SubscriptionResponseMapper.map(
            createSubscriptionsByOrganizationIdUseCase.execute(
                SubscriptionCommandFactory.map(kid, request))),
        HttpStatus.CREATED);
  }
}
