package sk.kosice.konto.kkmessageservice.message;

import java.util.UUID;
import sk.kosice.konto.kkmessageservice.domain.common.TimeProvider;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.BodyType;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ImmutableMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.recipient.entity.ImmutableRecipientEntity;
import sk.kosice.konto.kkmessageservice.domain.recipient.entity.RecipientEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.BaseOrganization;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.BaseTopic;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.ImmutableSubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.SubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.query.FindSubscriptionByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.ImmutableTopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.query.FindTopicByIdQuery;
import sk.kosice.konto.kkmessageservice.repository.message.JooqMessageRepository;
import sk.kosice.konto.kkmessageservice.repository.recipient.JooqRecipientRepository;
import sk.kosice.konto.kkmessageservice.repository.subscription.JooqSubscriptionRepository;
import sk.kosice.konto.kkmessageservice.repository.topic.JooqTopicRepository;

public interface MessageFeatureSpecTestSupport {

  default ImmutableTopicEntity prepareTopicData(UUID organizationId) {
    return ImmutableTopicEntity.builder()
        .id(UUID.randomUUID())
        .name("name")
        .description("description")
        .organizationId(organizationId)
        .organizationName("organizationName")
        .build();
  }

  default ImmutableSubscriptionEntity prepareSubscriptionData(UUID topicId, UUID kid) {
    return ImmutableSubscriptionEntity.builder()
        .id(UUID.randomUUID())
        .isEmailEnabled(false)
        .recipientKid(kid)
        .topic(new BaseTopic(topicId, null, null))
        .organization(new BaseOrganization(null, null))
        .build();
  }

  default TopicEntity persistAndFindTopic(JooqTopicRepository repository, UUID organizationId) {
    final var data = prepareTopicData(organizationId);
    repository.insert(data);
    return repository.findOne(FindTopicByIdQuery.of(data));
  }

  default SubscriptionEntity persistAndFindSubscription(
      JooqSubscriptionRepository repository, UUID topicId, UUID kid) {
    final var data = prepareSubscriptionData(topicId, kid);
    repository.insert(data);
    return repository.findOne(FindSubscriptionByIdQuery.of(data));
  }

  default ImmutableRecipientEntity prepareRecipientData() {
    return ImmutableRecipientEntity.builder()
        .kid(UUID.randomUUID())
        .email("ferko@kosice.sk")
        .build();
  }

  default ImmutableRecipientEntity prepareRecipientData(String kid) {
    return prepareRecipientData().withKid(UUID.fromString(kid));
  }

  default ImmutableMessageEntity prepareMessageData(UUID topicId) {
    return ImmutableMessageEntity.builder()
        .id(UUID.randomUUID())
        .senderName("sender")
        .title("title")
        .topicId(topicId)
        .topicName("none")
        .createdAt(TimeProvider.offsetNow())
        .organizationId(UUID.randomUUID())
        .body("body")
        .bodyType(BodyType.TEXT)
        .bodyShort("bodyShort")
        .isNotificationSend(true)
        .tags("[{\"a\":\"b\"},{\"c\":\"d\"}]")
        .recipientKid(UUID.randomUUID())
        .build();
  }

  default ImmutableMessageEntity prepareMessageData(UUID topicId, UUID recipientKid) {
    return prepareMessageData(topicId).withRecipientKid(recipientKid);
  }

  default ImmutableMessageEntity prepareMessageDataWithOrganization(
      UUID topicId, UUID organizationId) {
    return prepareMessageData(topicId).withOrganizationId(organizationId);
  }

  default RecipientEntity persistAndFindRecipient(
      JooqRecipientRepository repository, RecipientEntity insertData) {
    repository.insert(insertData);
    return repository.findOne(insertData.kid());
  }

  default MessageEntity persistAndFindMessage(
      JooqMessageRepository repository, MessageEntity insertData) {
    repository.insert(insertData);
    return repository.findOne(FindMessageByIdQuery.of(insertData));
  }
}
