'use client';

import { useTranslations } from 'next-intl';
import React, { useState } from 'react';
import { Loader, Table, TableRowValue, TableRow } from '@/lib/idsk';
import {
  TableSection,
  QueryHandler,
  TableList,
  Empty,
  Error,
  TableListItem,
  TableListValue,
  Pagination
} from '@/components';
import { useGetPets } from '@/hooks';
import { Pets as PetsDto } from '@/types';

const PAGE_SIZE = 10;

interface PetsProps {
  session: any;
  view: 'table' | 'list';
}

interface PetEntryProps {
  pet: PetsDto['pets'][0];
  view: 'list' | 'table';
}

const PetEntry: React.FC<PetEntryProps> = ({ pet, view }) => {
  const t = useTranslations('my_data');
  if (view === 'table') {
    return (
      <TableRow>
        <TableRowValue>{pet.name}</TableRowValue>
        <TableRowValue>{pet.breed}</TableRowValue>
        <TableRowValue>{pet.age}</TableRowValue>
        <TableRowValue align="right">{pet.evc}</TableRowValue>
      </TableRow>
    );
  }

  return (
    <TableListItem>
      <TableListValue label={t('name')}>{pet.name}</TableListValue>
      <TableListValue label={t('breed')}>{pet.breed}</TableListValue>
      <TableListValue label={t('age')}>{pet.age}</TableListValue>
      <TableListValue label={t('ecv')}>{pet.evc}</TableListValue>
    </TableListItem>
  );
};

const Pets: React.FC<PetsProps> = ({ session, view }) => {
  const [page, setPage] = useState(1);

  const t = useTranslations('my_data');
  const { data } = session;

  const userId = data?.user.id || '';
  const query = useGetPets(userId, page, PAGE_SIZE);

  return (
    <TableSection title={t('dogs')} description={t('dogs_description')}>
      <QueryHandler
        query={query}
        loading={
          <div className="px-6 py-12 flex items-center justify-center w-full">
            <Loader />
          </div>
        }
        error={<Error err={query.error} />}
      >
        {(petsDto: PetsDto) =>
          petsDto.pets.length > 0 ? (
            <>
              {view === 'table' ? (
                <Table
                  headRowBordered
                  headRow={
                    <>
                      <TableRowValue>{t('name')}</TableRowValue>
                      <TableRowValue>{t('breed')}</TableRowValue>
                      <TableRowValue>{t('age')}</TableRowValue>
                      <TableRowValue align="right">{t('ecv')}</TableRowValue>
                    </>
                  }
                >
                  {petsDto.pets.map((pet) => (
                    <PetEntry key={pet.id} pet={pet} view={view} />
                  ))}
                </Table>
              ) : (
                <TableList>
                  {petsDto.pets.map((pet) => (
                    <PetEntry key={pet.id} pet={pet} view={view} />
                  ))}
                </TableList>
              )}
              {petsDto.pagination.size > PAGE_SIZE && (
                <Pagination
                  className="ml-auto"
                  page={page}
                  pageSize={PAGE_SIZE}
                  totalCount={petsDto.pagination.size}
                  onPageChange={(selected) => setPage(selected)}
                />
              )}
            </>
          ) : (
            <Empty title={t('empty_dogs')} asSection />
          )
        }
      </QueryHandler>
    </TableSection>
  );
};

export default Pets;
