import React from 'react';
import { IUser, Result, MessageDetail as MessageDetailType } from '@/types';
import { axiosClient, baseUrl, headers } from '@/lib/axios/server';
import { MessageHeader, MessageBody, Error as ErrorUI } from '@/components';
import { getRequestError } from '@/utils/api/error';
import { validateMessageDetail } from '@/lib/zod';

interface MessageDetailProps {
  id: string;
  locale: string;
  user: IUser;
}

const getMessageDetail = async (id: string, userId: string): Promise<Result<MessageDetailType>> => {
  try {
    const response = await axiosClient.get(`${baseUrl('kkmessage')}/v1/kkmessage/recipients/${userId}/messages/${id}`, {
      headers: await headers()
    });

    return {
      data: response.data,
      error: undefined
    };
  } catch (error: unknown) {
    const requestError = getRequestError(error);
    return {
      data: undefined,
      error: requestError,
      status: requestError.status
    };
  }
};

const MessageDetail: React.FC<MessageDetailProps> = async ({ id, locale, user }) => {
  const { data, error } = await getMessageDetail(id, user.id);

  if (error) {
    return <ErrorUI err={error} />;
  }

  let detail = data as MessageDetailType;
  try {
    detail = validateMessageDetail(data) as MessageDetailType;
  } catch (err: unknown) {
    return <ErrorUI err={getRequestError(err)} />;
  }

  return (
    <div className="bg-white w-full border border-neutral-300 rounded-lg idsk-text-body">
      <MessageHeader
        title={detail.title}
        senderName={detail.senderName}
        createdAt={detail.createdAt}
        tags={detail.tags || []}
        locale={locale}
      />
      <MessageBody body={detail.body} actions={detail.actions || []} />
    </div>
  );
};

export default MessageDetail;
