'use client';

import React, { useMemo, useState } from 'react';
import { useTranslations } from 'next-intl';
import { AddIcon, SecondaryButton } from '@slovakia-kosice/idsk-react';
import {
  AddOrganizationDialog,
  Empty,
  Error,
  QueryHandler,
  Subscription,
  SubscriptionDeleteDialog,
  SubscriptionEditDialog,
  SubscriptionsSkeleton
} from '@/components';
import { Organization, Page } from '@/utils';
import useSubscriptions from './use-subscriptions';

interface CityOrganizationsProps {
  type: 'district' | 'organization';
  organizations: Organization[];
  onEdit: (item: Organization) => void;
  onDelete: (item: Organization) => void;
  onAdd: () => void;
}

const CityOrganizations: React.FC<CityOrganizationsProps> = ({ type, organizations, onEdit, onDelete, onAdd }) => {
  const t = useTranslations('profile_page');

  const isCityDistrict = type === 'district';
  const orgs = useMemo(() => organizations.filter((item) => item.isCityDistrict === isCityDistrict), [organizations]);

  return (
    <div className="flex flex-col gap-5">
      <div className="flex mb:flex-row flex-col gap-3 justify-between mb:items-end">
        <h4>{t(isCityDistrict ? 'town_districts' : 'organizations')}</h4>
        <SecondaryButton icon={<AddIcon />} onClick={onAdd}>
          {t(isCityDistrict ? 'add_town_district' : 'add_organization')}
        </SecondaryButton>
      </div>

      {orgs.length > 0 ? (
        <div className="border border-neutral-300 rounded-lg overflow-hidden">
          {orgs.map((item) => (
            <Subscription key={item.id} onEdit={() => onEdit(item)} onDelete={() => onDelete(item)} {...item} />
          ))}
        </div>
      ) : (
        <Empty
          title={t(isCityDistrict ? 'empty_subscriptions_districts' : 'empty_subscriptions_organizations')}
          asSection
        />
      )}
    </div>
  );
};

const SubscriptionsList: React.FC = () => {
  const { userId, query, onEdit, onDelete, selected, onCancelEdit } = useSubscriptions();

  const [addOrganization, setAddOrganization] = useState<'district' | 'organization' | null>(null);

  return (
    <>
      {selected?.type === 'edit' && (
        <SubscriptionEditDialog userId={userId} organization={selected} onClose={onCancelEdit} />
      )}
      {selected?.type === 'delete' && (
        <SubscriptionDeleteDialog userId={userId} organization={selected} onClose={onCancelEdit} />
      )}
      <QueryHandler query={query} error={<Error err={query.error} />} loading={<SubscriptionsSkeleton />}>
        {(data: Page<Organization>) => (
          <div className="flex flex-col gap-10">
            {addOrganization && (
              <AddOrganizationDialog
                type={addOrganization}
                userId={userId}
                subscribed={data.items}
                onClose={() => setAddOrganization(null)}
              />
            )}
            <CityOrganizations
              type="district"
              organizations={data.items}
              onEdit={onEdit}
              onDelete={onDelete}
              onAdd={() => setAddOrganization('district')}
            />
            <CityOrganizations
              type="organization"
              organizations={data.items}
              onEdit={onEdit}
              onDelete={onDelete}
              onAdd={() => setAddOrganization('organization')}
            />
          </div>
        )}
      </QueryHandler>
    </>
  );
};

export default SubscriptionsList;
