import axios, { AxiosHeaders, AxiosRequestConfig } from 'axios';
import { v4 as uuidv4 } from 'uuid';
import { NextRequest } from 'next/server';
import { auth } from '@/auth';
import { getRequestError } from '../../utils/api/error';

const version: string = process.env.version || '1.0.0';
const xAppId = process.env.API_HEADER_X_APP_ID || '';
const xDeviceId = process.env.API_HEADER_X_DEVICE_ID || '';
const xAppPlatform = process.env.API_HEADER_X_APP_PLATFORM || '';

// List of backend services
const services: Record<string, string | undefined> = {
  kkmessage: process.env.API_KKMESSAGE_BASE_URL || '',
  kk: process.env.API_KK_BASE_URL || ''
};

// finds service based on path: /v1/<servicename>/*
const getServiceName = (path: string) => {
  const match = path.match(/^\/v\d+\/([^/]+)/);
  return match ? match[1] : null;
};

export const baseUrl = (service: string): string => {
  const serviceUrl = services[service] as string;

  if (!serviceUrl) {
    throw new Error(`Base url for ${service} was not found.`);
  }

  return serviceUrl;
};

export const headers = async (otherHeaders?: Record<string, string>): Promise<AxiosHeaders> => {
  const token = (await auth())?.accessToken;

  return new AxiosHeaders({
    ...otherHeaders,
    correlationId: uuidv4(),
    Accept: 'application/json',
    'Content-Type': 'application/json',
    'X-APP-VERSION': version,
    'X-APP-ID': xAppId,
    'X-APP-PLATFORM': xAppPlatform,
    'X-DEVICE-ID': xDeviceId,
    ...(token ? { Authorization: `Bearer ${token}` } : {})
  });
};

const url = (request: NextRequest): string => {
  const path = request.nextUrl.pathname.replace('/api', '');
  const params = request.nextUrl.searchParams.toString();

  const serviceName = getServiceName(path);

  if (!serviceName) {
    throw new Error(`Unable to determine service name for http request from: ${path}`);
  }

  return `${baseUrl(serviceName)}${path}${params ? `?${params}` : ''}`;
};

const getConfig = async (request: NextRequest): Promise<AxiosRequestConfig> => {
  const config: Record<string, any> = {};

  const accept = request.headers.get('accept');
  config.responseType = accept?.includes('application/octet-stream') ? 'arraybuffer' : 'json';

  config.method = request.method;
  config.headers = {
    ...request.headers,
    ...(await headers())
  };
  config.url = url(request);

  if (request.body && request.headers.get('content-type') === 'application/json') {
    config.data = await request.json();
  }

  return config;
};

export const axiosClient = axios.create();

export const handler = async (request: NextRequest) => {
  const version = request.cookies.get('kk-version')?.value || '0.1.0';
  try {
    const config = await getConfig(request);
    config.headers!['X-APP-VERSION'] = version;

    const response = await axiosClient(config);

    if (config.responseType === 'arraybuffer') {
      const data = new Uint8Array(response.data);

      const disposition = response.headers['content-disposition'];

      return new Response(data, {
        status: 200,
        headers: {
          'Content-Type': 'application/octet-stream',
          'Content-Disposition': disposition
        }
      });
    }

    return Response.json(response.data);
  } catch (err: unknown) {
    const error = getRequestError(err);
    return Response.json(error, { status: error.status });
  }
};
