package sk.kosice.konto.kkmessageservice.repository.rsql.jooq.builder;

import cz.jirutka.rsql.parser.ast.ComparisonNode;
import cz.jirutka.rsql.parser.ast.ComparisonOperator;
import cz.jirutka.rsql.parser.ast.LogicalNode;
import cz.jirutka.rsql.parser.ast.RSQLOperators;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.jooq.Condition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListFilteringOperators;
import sk.kosice.konto.kkmessageservice.repository.rsql.RsqlErrorCode;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqColumnInfo;

public class JooqRsqlConditionBuilder extends JooqRsqlWithListingMetadataBuilder<Condition> {
  private static final Logger log = LoggerFactory.getLogger(JooqRsqlConditionBuilder.class);

  public static final String NULL_VALUE = "null";

  @Override
  public Condition build(LogicalNode logicalNode) {
    final List<Condition> childConditions =
        logicalNode.getChildren().stream().map(this::build).collect(Collectors.toList());

    switch (logicalNode.getOperator()) {
      case AND:
        return childConditions.stream().reduce(Condition::and).get();
      case OR:
        return childConditions.stream().reduce(Condition::or).get();
      default:
        throw new IllegalArgumentException("Unknown logical operator " + logicalNode.getOperator());
    }
  }

  @Override
  public Condition build(ComparisonNode comparisonNode) {

    final String property = comparisonNode.getSelector();
    final ComparisonOperator operator = comparisonNode.getOperator();
    final List<String> arguments = comparisonNode.getArguments();

    final Optional<JooqColumnInfo<?>> maybe = getMetadata().findByName(property);

    if (maybe.isEmpty()) {
      throw new IllegalArgumentException("Unknown property: " + property);
    }

    final JooqColumnInfo<Object> columnInfo = (JooqColumnInfo<Object>) maybe.get();

    if (operator.equals(RSQLOperators.EQUAL)) {
      return buildEqualCondition(columnInfo, arguments.get(0));
    } else if (operator.equals(RSQLOperators.NOT_EQUAL)) {
      return buildNotEqualCondition(columnInfo, arguments.get(0));
    } else if (operator.equals(RSQLOperators.GREATER_THAN)) {
      return columnInfo.getField().greaterThan(columnInfo.getConverter().from(arguments.get(0)));
    } else if (operator.equals(RSQLOperators.GREATER_THAN_OR_EQUAL)) {
      return columnInfo.getField().greaterOrEqual(columnInfo.getConverter().from(arguments.get(0)));
    } else if (operator.equals(RSQLOperators.LESS_THAN)) {
      return columnInfo.getField().lessThan(columnInfo.getConverter().from(arguments.get(0)));
    } else if (operator.equals(RSQLOperators.LESS_THAN_OR_EQUAL)) {
      return columnInfo.getField().lessOrEqual(columnInfo.getConverter().from(arguments.get(0)));
    } else if (operator.equals(RSQLOperators.IN)) {
      return columnInfo
          .getField()
          .in(
              arguments.stream()
                  .map(a -> columnInfo.getConverter().from(a))
                  .collect(Collectors.toList()));
    } else if (operator.equals(RSQLOperators.NOT_IN)) {
      return columnInfo
          .getField()
          .notIn(
              arguments.stream()
                  .map(a -> columnInfo.getConverter().from(a))
                  .collect(Collectors.toList()));
    } else if (operator.equals(ListFilteringOperators.LIKE)) {
      if (!columnInfo.getConverter().isAccessibleFor(String.class)) {
        log.error(
            "Not supported operator: {} for field {}, because of its data type: {}",
            operator,
            property,
            columnInfo.getField().getType().getName());
        throw RsqlErrorCode.INVALID_RSQL_OPERATOR
            .createError(operator, property)
            .convertToException();
      }
      return columnInfo
          .getField()
          .likeIgnoreCase(
              arguments
                  .get(0)
                  .replaceFirst("^\\s*(\\*)?\\s*(.*?)\\s*(\\*)?\\s*$", "$1$2$3")
                  .replace('*', '%'));
    } else {
      throw new IllegalArgumentException("Unknown operator: " + operator);
    }
  }

  private Condition buildEqualCondition(JooqColumnInfo<Object> fieldInfo, String argument) {
    if (NULL_VALUE.equals(argument)) {
      return fieldInfo.getField().isNull();
    }
    return fieldInfo.getField().eq(fieldInfo.getConverter().from(argument));
  }

  private Condition buildNotEqualCondition(JooqColumnInfo<Object> fieldInfo, String argument) {
    if (NULL_VALUE.equals(argument)) {
      return fieldInfo.getField().isNotNull();
    }
    return fieldInfo.getField().notEqual(fieldInfo.getConverter().from(argument));
  }
}
