package sk.kosice.konto.kkmessageservice.repository.subscription;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ImmutableListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.PageSize;
import sk.kosice.konto.kkmessageservice.domain.common.listing.SubscriptionListingAttribute;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.ListOfSubscriptions;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.SubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.query.ImmutableSubscriptionListingQuery;

public class JooqSubscriptionRepositoryListTest extends JooqSubscriptionRepositoryTest {

  @Test
  public void thatEmptySubscriptionListCanBeRetrieved() {
    final ListOfSubscriptions subscriptions =
        listSubscriptions(subscriptionRepository, UUID.randomUUID());
    assertThat(subscriptions.items()).isEmpty();
    assertThat(subscriptions.totalCount()).isZero();
  }

  @Test
  public void thatSubscriptionListCanBeRetrieved() {
    final List<SubscriptionEntity> subscriptions = createSubscriptionsForFilteringAndSorting();
    final ListOfSubscriptions result =
        listSubscriptions(subscriptionRepository, subscriptions.getFirst().recipientKid());

    assertThat(result.items()).isNotEmpty();
    assertThat(result.totalCount()).isEqualTo(3);
  }

  @Test
  public void thatSubscriptionListCanBeFilteredByTopicName() {
    final List<SubscriptionEntity> subscriptions = createSubscriptionsForFilteringAndSorting();
    findWithFilter(
        subscriptions.getFirst().recipientKid(),
        "topicName==\"" + subscriptions.get(0).topic().name() + "\"",
        1);
  }

  @Test
  public void thatSubscriptionListCanBeSortedByTopicName() {
    final List<SubscriptionEntity> subscriptions = createSubscriptionsForFilteringAndSorting();

    findSorted(
        subscriptions.getFirst().recipientKid(),
        new SubscriptionEntity[] {subscriptions.get(0), subscriptions.get(1), subscriptions.get(2)},
        ImmutableListOrdering.of(SubscriptionListingAttribute.TOPIC_NAME, true));

    findSorted(
        subscriptions.getFirst().recipientKid(),
        new SubscriptionEntity[] {subscriptions.get(2), subscriptions.get(1), subscriptions.get(0)},
        ImmutableListOrdering.of(SubscriptionListingAttribute.TOPIC_NAME, false));
  }

  @Test
  public void thatSubscriptionListCanBePaginated() {
    final List<SubscriptionEntity> subscriptions = createSubscriptionsForFilteringAndSorting();

    findPage(
        subscriptions.getFirst().recipientKid(),
        1,
        1,
        ImmutableListOrdering.of(SubscriptionListingAttribute.TOPIC_NAME, true),
        subscriptions.get(0));

    findPage(
        subscriptions.getFirst().recipientKid(),
        2,
        2,
        ImmutableListOrdering.of(SubscriptionListingAttribute.TOPIC_NAME, true),
        subscriptions.get(2));

    findPage(
        subscriptions.getFirst().recipientKid(),
        3,
        1,
        ImmutableListOrdering.of(SubscriptionListingAttribute.TOPIC_NAME, false),
        subscriptions.get(2),
        subscriptions.get(1),
        subscriptions.get(0));
  }

  private List<SubscriptionEntity> createSubscriptionsForFilteringAndSorting() {
    final List<UUID> subscriptions = prepareTopics();
    final UUID recipientKid = prepareRecipient();

    prepareSubscriptions(recipientKid, subscriptions.toArray(UUID[]::new));

    return findSubscriptions(subscriptionRepository, recipientKid);
  }

  private void findWithFilter(UUID kid, String rsqlQuery, int expectedNumOfResults) {
    final ListOfSubscriptions listOfSubscriptions =
        subscriptionRepository.list(
            ImmutableSubscriptionListingQuery.builder()
                .pageSize(PageSize.of(100, 100))
                .recipientKid(kid)
                .addAllOrderings(Collections.emptyList())
                .rsqlQuery(Optional.of(rsqlParser.parse(rsqlQuery)))
                .build());

    assertThat(listOfSubscriptions.totalCount()).isEqualTo(expectedNumOfResults);
  }

  @SafeVarargs
  private void findSorted(
      UUID kid,
      SubscriptionEntity[] expectedOrder,
      ListOrdering<SubscriptionListingAttribute>... orderings) {
    final ListOfSubscriptions listOfSubscriptions =
        subscriptionRepository.list(
            ImmutableSubscriptionListingQuery.builder()
                .pageSize(PageSize.of(100, 100))
                .recipientKid(kid)
                .addAllOrderings(List.of(orderings))
                .rsqlQuery(Optional.empty())
                .build());

    assertThat(listOfSubscriptions.items()).containsExactly(expectedOrder);
  }

  private void findPage(
      UUID kid,
      int pageSize,
      int pageNumber,
      ListOrdering<SubscriptionListingAttribute> ordering,
      SubscriptionEntity... expectedOrder) {
    final ListOfSubscriptions listOfSubscriptions =
        subscriptionRepository.list(
            ImmutableSubscriptionListingQuery.builder()
                .pageSize(PageSize.of(100, pageSize))
                .recipientKid(kid)
                .rsqlQuery(Optional.empty())
                .page(pageNumber)
                .addAllOrderings(List.of(ordering))
                .build());

    assertThat(listOfSubscriptions.items()).containsExactly(expectedOrder);
  }
}
