package sk.kosice.konto.kkmessageservice.repository.topic;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static sk.kosice.konto.kkmessageservice.domain.topic.error.TopicErrorCode.TOPIC_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.repository.topic.error.constraint.TopicConstraintErrorCode.TOPIC_WITH_ID_ALREADY_EXISTS;
import static sk.kosice.konto.kkmessageservice.repository.topic.error.constraint.TopicConstraintErrorCode.TOPIC_WITH_NAME_ALREADY_EXISTS_IN_ORGANIZATION;

import java.util.UUID;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.ImmutableTopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.query.FindTopicByIdQuery;

public class JooqTopicRepositoryFindTest extends JooqTopicRepositoryTest {

  @Test
  public void thatTopicCanBeCreatedAndSelected() {
    final var insertData = prepareTopicData();
    persistTopic(topicRepository, insertData);

    final var selected = findTopic(topicRepository, FindTopicByIdQuery.of(insertData));
    assertThat(selected.id()).isEqualTo(insertData.id());
    assertThat(selected.name()).isEqualTo(insertData.name());
    assertThat(selected.description()).isEqualTo(insertData.description());
    assertThat(selected.organizationId()).isEqualTo(insertData.organizationId());
  }

  @Test
  public void thatExistingTopicCannotBeInserted() {
    final TopicEntity insertData = prepareTopicData();
    persistTopic(topicRepository, insertData);

    final BusinessException exception =
        assertThrows(BusinessException.class, () -> persistTopic(topicRepository, insertData));

    assertThat(exception.getError().code().toString())
        .isEqualTo(TOPIC_WITH_ID_ALREADY_EXISTS.name());
  }

  @Test
  public void thatDuplicatedNameOfTopicCannotBeInserted() {
    final TopicEntity insertData = prepareTopicData();
    final TopicEntity insertDuplicatedNameData =
        ImmutableTopicEntity.copyOf(insertData).withId(UUID.randomUUID());
    persistTopic(topicRepository, insertData);

    final BusinessException exception =
        assertThrows(
            BusinessException.class, () -> persistTopic(topicRepository, insertDuplicatedNameData));

    assertThat(exception.getError().code().toString())
        .isEqualTo(TOPIC_WITH_NAME_ALREADY_EXISTS_IN_ORGANIZATION.name());
  }

  @Test
  public void thatNotExistingTopicCannotBeSelected() {
    final BusinessException exception =
        assertThrows(
            BusinessException.class,
            () ->
                findTopic(
                    topicRepository, FindTopicByIdQuery.of(UUID.randomUUID(), UUID.randomUUID())));

    assertThat(exception.getError().code().toString()).isEqualTo(TOPIC_DOES_NOT_EXIST.toString());
  }
}
