package sk.kosice.konto.kkmessageservice.subscription;

import static org.assertj.core.api.Assertions.assertThat;
import static sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser.ASCENDING;
import static sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser.DESCENDING;

import java.util.List;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.domain.common.listing.SubscriptionListingAttribute;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionListResponse;

public class SubscriptionListingFeatureSpec extends SubscriptionFeatureSpec {

  @Test
  public void thatEmptySubscriptionListCanBeRetrieved() {
    prepareRecipient(UUID.fromString(EXTENSION_KID_CLAIM));
    final SubscriptionListResponse subscriptions =
        listSubscriptions(
            HttpStatus.OK, UUID.fromString(EXTENSION_KID_CLAIM), SubscriptionListResponse.class);
    assertThat(subscriptions.items()).isEmpty();
    assertThat(subscriptions.totalCount()).isZero();
  }

  @Test
  public void thatSubscriptionListCanBeRetrieved() {
    final List<SubscriptionDetailResponse> subscriptions =
        createSubscriptionsForFilteringAndSorting(UUID.fromString(EXTENSION_KID_CLAIM));
    final SubscriptionListResponse result =
        listSubscriptions(
            HttpStatus.OK,
            subscriptions.getFirst().getRecipientKid(),
            SubscriptionListResponse.class);

    assertThat(result.items()).isNotEmpty();
    assertThat(result.totalCount()).isEqualTo(3);
  }

  @Test
  public void thatSubscriptionListCanBeFilteredByTopicName() {
    final List<SubscriptionDetailResponse> subscriptions =
        createSubscriptionsForFilteringAndSorting(UUID.fromString(EXTENSION_KID_CLAIM));
    final String filter =
        String.format(
            "?filter=%s==%s",
            SubscriptionListingAttribute.TOPIC_NAME.apiName(),
            subscriptions.get(0).getTopic().name());

    checkFiltering(UUID.fromString(EXTENSION_KID_CLAIM), filter, 1, subscriptions.get(0));
  }

  @Test
  public void thatSubscriptionListCanBeSortedByTopicName() {
    final List<SubscriptionDetailResponse> subscriptions =
        createSubscriptionsForFilteringAndSorting(UUID.fromString(EXTENSION_KID_CLAIM));

    checkSorting(
        UUID.fromString(EXTENSION_KID_CLAIM),
        SubscriptionListingAttribute.TOPIC_NAME.apiName() + ASCENDING,
        subscriptions.size(),
        subscriptions.get(0),
        subscriptions.get(1),
        subscriptions.get(2));

    checkSorting(
        UUID.fromString(EXTENSION_KID_CLAIM),
        SubscriptionListingAttribute.TOPIC_NAME.apiName() + DESCENDING,
        subscriptions.size(),
        subscriptions.get(2),
        subscriptions.get(1),
        subscriptions.get(0));
  }

  private List<SubscriptionDetailResponse> createSubscriptionsForFilteringAndSorting(
      UUID recipientKid) {
    final List<UUID> subscriptions = prepareTopics();
    prepareRecipient(recipientKid);
    prepareSubscriptions(recipientKid, subscriptions.toArray(UUID[]::new));

    return listSubscriptions(HttpStatus.OK, recipientKid, SubscriptionListResponse.class).items();
  }
}
