import { useMutation, useQueryClient } from '@tanstack/react-query';
import axios from 'axios';
import { useTranslations } from 'next-intl';
import { ChangeEvent, useState } from 'react';
import { z } from 'zod';
import { ContactType, IUser, Profile, unwrap } from '@/utils';
import { useNotifications } from '@/hooks';

const emailValidationSchema = z.string().email();
const phoneValidationSchema = z.string().regex(/^\+421\s?\d{9}$/);

function useContact(user: IUser | undefined, type: ContactType) {
  const queryClient = useQueryClient();
  const notify = useNotifications();
  const t = useTranslations('my_data');

  const [value, setValue] = useState<string>('');
  const [error, setError] = useState<boolean>(false);

  const getMutationOptions = (errorMessage: string) => ({
    onError: (_err: any, _mutationData: any, context: any) => {
      queryClient.setQueryData(['profile'], context.prevProfile);

      notify({
        message: t(errorMessage),
        variant: 'warning'
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['profile'] });
    }
  });

  const add = useMutation({
    mutationKey: ['contact'],
    mutationFn: (mutationData: any) => unwrap(axios.post(`/api/v1/kk/profiles/${user?.id}/params`, mutationData)),
    onMutate: async (mutationData: any) => {
      await queryClient.cancelQueries({ queryKey: ['profile'] });
      const prevProfile: Profile | undefined = queryClient.getQueryData(['profile']);

      if (prevProfile) {
        const key = type === 'email' ? 'emails' : 'phones';
        const newContacts = [...prevProfile.identity[key], mutationData.value];

        queryClient.setQueryData(['profile'], (old: Profile) => ({
          ...old,
          identity: {
            ...old.identity,
            [key]: newContacts
          }
        }));
      }

      notify(t('contact_success_notification'));

      setValue('');
      return { prevProfile };
    },
    ...getMutationOptions('contact_error_notification')
  });

  const remove = useMutation({
    mutationKey: ['contact'],
    mutationFn: (mutationData: any) =>
      unwrap(
        axios.delete(`/api/v1/kk/profiles/${user?.id}/params`, {
          data: mutationData
        })
      ),

    onMutate: async (mutationData: any) => {
      await queryClient.cancelQueries({ queryKey: ['profile'] });
      const prevProfile: Profile | undefined = queryClient.getQueryData(['profile']);

      if (prevProfile) {
        const key = type === 'email' ? 'emails' : 'phones';
        const newContacts = prevProfile.identity[key].filter((contact) => contact !== mutationData.value);

        queryClient.setQueryData(['profile'], (old: Profile) => ({
          ...old,
          identity: {
            ...old.identity,
            [key]: newContacts
          }
        }));
      }

      notify(t('contact_delete_success_notification'));

      return { prevProfile };
    },
    ...getMutationOptions('contact_delete_error_notification')
  });

  const handleChange = async (e: ChangeEvent<HTMLInputElement>) => {
    const val = e.target.value;
    setValue(val);

    if (val === '') setError(false);

    try {
      if (type === 'email') {
        emailValidationSchema.parse(val);
      } else {
        phoneValidationSchema.parse(val);
      }

      if (error) {
        setError(false);
      }
    } catch (err: unknown) {
      console.error(err);

      if (!error) {
        setError(true);
      }
    }
  };

  return {
    add,
    remove,
    value,
    error,
    handleChange
  };
}

export default useContact;
