'use client';

import React from 'react';
import { useTranslations } from 'next-intl';
import { useState, useEffect, useMemo, useCallback } from 'react';
import { useQueryClient } from '@tanstack/react-query';
import { useGetOrganizations, useNotifications, useUpdateSubscriptions } from '@/hooks';
import { Checkbox, Dialog, Loader, PrimaryButton, SearchBar, SecondaryButton } from '@/lib/idsk';
import { Organization, Page } from '@/types';
import { Empty, Error, QueryHandler } from '@/components/core';

interface AddOrganizationDialogProps {
  type: 'district' | 'organization';
  userId: string;
  onClose: () => void;
  subscribed: Organization[];
}

interface AddOrganizationDialogListProps {
  type: 'district' | 'organization';
  allOrganizations: Organization[];
  subscribedOrganizations: Organization[];
  onChange: (changes: number, items: SelectableOrganization[]) => void;
}

type Selectable<T> = T & { selected: boolean };
type Searchable<T> = T & { visible: boolean };
type Disabled<T> = T & { disabled: boolean };

type SelectableOrganization = Disabled<Searchable<Selectable<Organization>>>;

const getInitialOrganizations = (
  type: 'district' | 'organization',
  all: Organization[],
  subscribed: Organization[]
) => {
  const isCityDistrict = type === 'district';
  return all
    .filter((item) => item.isCityDistrict === isCityDistrict)
    .map((item) => ({
      ...item,
      selected: !!subscribed.find((current) => current.id === item.id),
      disabled: !!subscribed.find((current) => current.id === item.id),
      visible: true
    }));
};

const AddOrganizationDialogList: React.FC<AddOrganizationDialogListProps> = ({
  type,
  allOrganizations,
  subscribedOrganizations,
  onChange
}) => {
  const initialItems = useMemo(
    () => getInitialOrganizations(type, allOrganizations, subscribedOrganizations),
    [allOrganizations, subscribedOrganizations]
  );

  const [search, setSearch] = useState<string>('');
  const [current, setCurrent] = useState<SelectableOrganization[]>(initialItems);
  const [changes, setChanges] = useState<number>(0);

  const t = useTranslations('profile_page');

  useEffect(() => {
    const organizations = current.map((item) => ({
      ...item,
      visible: item.name.toLowerCase().trim().includes(search.toLowerCase().trim())
    }));
    setCurrent(organizations);
  }, [search]);

  useEffect(() => {
    onChange(changes, current);
  }, [changes]);

  const handleChecked = (organization: SelectableOrganization, checked: boolean) => {
    const target = initialItems.find((item) => item.id === organization.id);
    if (target) {
      const increment = target.selected === checked ? -1 : 1;
      setChanges((p) => p + increment);
    }
    setCurrent((p) => p.map((item) => ({ ...item, selected: item.id === organization.id ? checked : item.selected })));
  };

  return (
    <>
      <SearchBar
        value={search}
        onChange={(e) => setSearch(e.target.value)}
        onCancel={() => setSearch('')}
        showCancelButton={!!search}
        fullWidth
        searchbarSize="medium"
      />
      <div className="w-full flex flex-col gap-2.5">
        {current.length > 0 ? (
          current.map((item) =>
            item.visible ? (
              <Checkbox
                key={item.id}
                label={item.name}
                checked={item.selected}
                disabled={item.disabled}
                onChange={(e) => handleChecked(item, e.target.checked)}
              />
            ) : null
          )
        ) : (
          <Empty title={t(`empty_${type}`)} />
        )}
      </div>
    </>
  );
};

const AddOrganizationDialog: React.FC<AddOrganizationDialogProps> = ({ type, subscribed, userId, onClose }) => {
  const [changes, setChanges] = useState<{ changes: number; items: SelectableOrganization[] }>({
    changes: 0,
    items: []
  });

  const t = useTranslations('profile_page');
  const notify = useNotifications();

  const queryClient = useQueryClient();

  const query = useGetOrganizations();
  const mutation = useUpdateSubscriptions({
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['subscriptions'] });
      notify(t(`subscriptions_success_${type}`));
      onClose();
    },
    onError: () => {
      notify({ message: t('subscriptions_error'), variant: 'warning' });
    }
  });

  const items = useMemo(() => {
    const isCityDistrict = type === 'district';
    return subscribed.filter((item) => item.isCityDistrict === isCityDistrict);
  }, [type, subscribed]);

  const handleChange = useCallback((changesCount: number, changesItems: SelectableOrganization[]) => {
    setChanges({ changes: changesCount, items: changesItems });
  }, []);

  const handleSaveChanges = () => {
    const data = changes.items.filter((item) => item.selected).map((item) => item.id);
    mutation.mutate({ userId, items: data } as any);
  };

  return (
    <Dialog
      opened
      className="custom-dialog dialog--large"
      title={t(`add_dialog_title_${type}`)}
      primaryButton={
        <PrimaryButton
          className="h-fit"
          onClick={handleSaveChanges}
          disabled={changes.changes === 0 || mutation.isPending}
        >
          {t('confirm')}
        </PrimaryButton>
      }
      secondaryButton={
        <div className="relative">
          {mutation.isPending && <Loader spinnerClassName="w-12 absolute -left-16 -top-2.5" />}
          <SecondaryButton onClick={onClose}>{t('close')}</SecondaryButton>
        </div>
      }
      toggleOpened={onClose}
    >
      <div className="flex flex-col gap-5 w-full">
        <div className="flex flex-col gap-2.5">
          <h4>{t('pick_from_list')}</h4>
        </div>

        <QueryHandler
          query={query}
          loading={
            <div className="w-full p-12 flex items-center justify-center">
              <Loader />
            </div>
          }
          error={<Error err={query.error} />}
        >
          {(data: Page<Organization>) => (
            <AddOrganizationDialogList
              type={type}
              allOrganizations={data.items}
              subscribedOrganizations={items}
              onChange={handleChange}
            />
          )}
        </QueryHandler>
      </div>
    </Dialog>
  );
};

export default AddOrganizationDialog;
