import { NextFetchEvent, NextRequest, NextResponse } from 'next/server';
import { CustomMiddleware } from './chain';
import { auth } from '@/auth';
import { cookies } from 'next/headers';
import { DEFAULT_LOCALE } from '@/i18n/routing';

const PUBLIC_ROUTES = ['/uvod', '/odhlasenie', '/signin', '/error', '/cookies'];
const LOGIN_ROUTE = '/uvod';

const isPublicRoute = (path: string, publicRoutes: string[]) => {
  for (const page of publicRoutes) {
    if (path.startsWith(page)) {
      return true;
    }
  }
  return false;
};

export function withAuthMiddleware(middleware: CustomMiddleware): CustomMiddleware {
  return async (request: NextRequest, event: NextFetchEvent, response: NextResponse) => {
    const session = await auth();
    const isLogged = !!session && !session.error;

    const store = await cookies();
    const locale = store.get('NEXT_LOCALE')?.value || DEFAULT_LOCALE;

    const { pathname } = request.nextUrl;
    let route = pathname;

    if (route.startsWith(`/${locale}/`) || route === `/${locale}`) {
      route = route.substring(`/${locale}`.length);
    }

    // Unauthorized user cannot access private page
    if (!isLogged && !isPublicRoute(route, PUBLIC_ROUTES)) {
      return NextResponse.redirect(new URL(`/${locale}${LOGIN_ROUTE}`, request.url));
    }

    // Logged in user, should not stay on login page
    if (isLogged && route === LOGIN_ROUTE) {
      return NextResponse.redirect(new URL(`/${locale}`, request.url));
    }

    return middleware(request, event, response);
  };
}
