'use client';

import axios from 'axios';
import { api, NewMessage, Topic, unwrap } from '@/utils';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { useSession } from 'next-auth/react';
import { useState } from 'react';

type PostData = {
  url: string;
} & NewMessage;

const useNewMessage = (onSuccess: () => void) => {
  const [message, setMessage] = useState<string>('');
  const [filter, setFilter] = useState<string>('');
  const [previewIsOpen, setPreviewIsOpen] = useState(false);
  const [topic, setTopic] = useState<Topic | null>(null);
  const [tags, setTags] = useState<any[]>([]);

  const queryClient = useQueryClient();

  const { data } = useSession();
  const user = data?.user;

  const mutation = useMutation({
    mutationKey: ['newMessage'],
    mutationFn: (data: PostData) => {
      const { url, ...body } = data;
      return unwrap(axios.post(`/api${url}`, body));
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['messages'] });
      onSuccess();
    }
  });

  const query = useQuery({
    queryKey: ['topics', filter],
    queryFn: () =>
      unwrap(
        axios.get(`/api${api.topics(data?.user.organization?.id || '')}`, {
          params: {
            'page-size': 50,
            filter: `name=like='*${filter}*'`,
            sort: 'name;'
          }
        })
      )
  });

  const send = (formData: FormData) => {
    const postData = {
      title: formData.get('subject') as string,
      body: message,
      bodyShort: message.substring(0, 20),
      bodyType: 'HTML',
      topicId: topic?.id as string,
      tags: tags.map((tag) => ({ value: tag.value, color: tag.color })),
      actions: []
    };

    const kid = formData.get('kid');

    if (!!kid) {
      const id = kid as string;
      mutation.mutate({ ...postData, url: api.messagesKid(id) });
      return;
    }

    const id = user?.organization?.id as string;
    mutation.mutate({ ...postData, url: api.messages(id) });
  };

  return {
    mutation,
    query,
    send,
    user,
    message: {
      set: setMessage,
      get: () => message
    },
    filter: {
      set: setFilter,
      get: () => filter
    },
    topic: {
      set: setTopic,
      get: () => topic
    },
    tags: {
      set: setTags,
      get: () => tags
    },
    preview: {
      isOpen: () => previewIsOpen,
      toggleOpen: (value?: boolean) => setPreviewIsOpen((p) => (value ? value : !p))
    }
  };
};

export default useNewMessage;
