package sk.kosice.konto.kknotificationservice.repository.config;

import java.util.concurrent.Executors;
import javax.sql.DataSource;
import org.jooq.DSLContext;
import org.jooq.SQLDialect;
import org.jooq.conf.MappedSchema;
import org.jooq.conf.RenderMapping;
import org.jooq.conf.RenderNameCase;
import org.jooq.conf.RenderQuotedNames;
import org.jooq.conf.Settings;
import org.jooq.impl.DSL;
import org.jooq.impl.DataSourceConnectionProvider;
import org.jooq.impl.DefaultConfiguration;
import org.jooq.impl.ThreadLocalTransactionProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.jdbc.DataSourceBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.jdbc.datasource.TransactionAwareDataSourceProxy;
import sk.kosice.konto.kknotificationservice.domain.shared.utils.SecretUtils;

@Configuration
public class JooqConfiguration {

  public static final String KKMESSAGE_SERVICE_DS = "kkmessage-service-data-source";
  public static final String KKMESSAGE_SERVICE_DSL = "kkmessage-service-dsl-context";

  public static final String KKMESSAGE_SERVICE_TX_DS_PROXY = "kkmessage-tx-aware-data-source-proxy";
  public static final String KKMESSAGE_SERVICE_DS_CONN_PROVIDER =
      "kkmessage-data-source-connection-provider";

  @Value("${spring.jooq.sql-dialect}")
  private String jooqDialect;

  @Value("${spring.jooq.schema-name}")
  private String schemaName;

  @Value("${spring.datasource.jdbc-url}")
  private String url;

  @Value("${spring.datasource.username}")
  private String username;

  @Value("${spring.datasource.password}")
  private String password;

  @Bean(name = KKMESSAGE_SERVICE_DS)
  @Primary
  DataSource dataSource() {
    return DataSourceBuilder.create()
        .url(url)
        .username(username)
        .password(SecretUtils.resolveSecretValue(password))
        .build();
  }

  @Bean(name = KKMESSAGE_SERVICE_TX_DS_PROXY)
  TransactionAwareDataSourceProxy transactionAwareDataSourceProxy(DataSource dataSource) {
    return new TransactionAwareDataSourceProxy(dataSource);
  }

  @Bean(name = KKMESSAGE_SERVICE_DS_CONN_PROVIDER)
  DataSourceConnectionProvider dataSourceConnectionProvider(
      @Qualifier(KKMESSAGE_SERVICE_TX_DS_PROXY) TransactionAwareDataSourceProxy proxy) {
    return new DataSourceConnectionProvider(proxy);
  }

  @Bean(name = KKMESSAGE_SERVICE_DSL)
  DSLContext dsl(
      @Qualifier(KKMESSAGE_SERVICE_DS_CONN_PROVIDER) DataSourceConnectionProvider provider) {

    final Settings settings =
        new Settings()
            .withRenderQuotedNames(RenderQuotedNames.EXPLICIT_DEFAULT_UNQUOTED)
            .withRenderNameCase(RenderNameCase.UPPER)
            .withRenderSchema(true)
            .withRenderCatalog(false)
            .withRenderMapping(
                new RenderMapping()
                    .withSchemata(
                        new MappedSchema()
                            .withInput("DB_SCHEMA_REPLACEABLE")
                            .withOutput(schemaName)));

    return DSL.using(
        new DefaultConfiguration()
            .set(SQLDialect.valueOf(jooqDialect))
            .set(settings)
            .set(Executors.newCachedThreadPool())
            .set(new ThreadLocalTransactionProvider(provider)));
  }
}
